"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const super_easy_validator_1 = require("super-easy-validator");
const constants_1 = __importDefault(require("../../utils/constants"));
function getChats(req, res, next) {
    try {
        const rules = {
            limit: 'optional|string|natural',
            page: 'optional|string|natural',
            search: 'optional|string',
            group: 'optional|string|boolean',
            meetMeOnly: 'optional|string|boolean',
        };
        const { errors } = (0, super_easy_validator_1.validate)(rules, req.query);
        if (errors) {
            return res.status(400).json({ message: errors[0] });
        }
        return next();
    }
    catch (error) {
        console.log(error);
        return res.status(500).json({ message: 'server error' });
    }
}
function getExploreChats(req, res, next) {
    try {
        const rules = {
            limit: 'optional|string|natural',
            page: 'optional|string|natural',
            search: 'optional|string',
            category: 'optional|string|min:1|max:20',
            reelsLimit: 'optional|string|natural',
        };
        const { errors } = (0, super_easy_validator_1.validate)(rules, req.query);
        if (errors) {
            return res.status(400).json({ message: errors[0] });
        }
        return next();
    }
    catch (error) {
        console.log(error);
        return res.status(500).json({ message: 'server error' });
    }
}
function getChatFiles(req, res, next) {
    try {
        const rules = {
            limit: 'optional|string|natural',
            page: 'optional|string|natural',
            tag: 'optional|string|min:1|max:15',
            type: 'optional|string|enums:video,audio,image,document,other'
        };
        const { errors } = (0, super_easy_validator_1.validate)(rules, req.query);
        if (errors) {
            return res.status(400).json({ message: errors[0] });
        }
        return next();
    }
    catch (error) {
        console.log(error);
        return res.status(500).json({ message: 'server error' });
    }
}
function postChat(req, res, next) {
    try {
        const rules = {
            user: 'optional|mongoid',
            meetMe: 'optional|mongoid',
            isGroup: 'optional|boolean',
            category: 'optional|string|min:1|max:20',
            name: 'optional|string|min:3|max:30',
            participants: 'optional|array|min:1|max:10|arrayof:mongoid',
            restricted: 'optional|boolean',
            socket: `optional|${constants_1.default.socketRegex}`
        };
        const { errors } = (0, super_easy_validator_1.validate)(rules, req.body);
        if (errors) {
            return res.status(400).json({ message: errors[0] });
        }
        if (req.body.isGroup) {
            if (req.body.name) {
                req.body.name = req.body.name.trim();
            }
            const rules = {
                name: 'string|min:3|max:25',
                participants: 'array|min:1|arrayof:mongoid',
                category: 'string|min:1|max:20',
            };
            const { errors } = (0, super_easy_validator_1.validate)(rules, req.body);
            if (errors) {
                return res.status(400).json({ message: errors[0] });
            }
        }
        if (!req.body.isGroup && !req.body.user) {
            return res.status(400).json({ message: 'user is required in one-to-one chat' });
        }
        return next();
    }
    catch (error) {
        console.log(error);
        return res.status(500).json({ message: 'server error' });
    }
}
function postChatParticipants(req, res, next) {
    try {
        const rules = {
            id: 'mongoid',
            participants: 'array|min:1|max:10|arrayof:mongoid'
        };
        const { errors } = (0, super_easy_validator_1.validate)(rules, Object.assign(Object.assign({}, req.body), req.params));
        if (errors) {
            return res.status(400).json({ message: errors[0] });
        }
        return next();
    }
    catch (error) {
        console.log(error);
        return res.status(500).json({ message: 'server error' });
    }
}
function postChatParticipant(req, res, next) {
    try {
        const rules = {
            id: 'mongoid',
            socket: `optional|${constants_1.default.socketRegex}`
        };
        const { errors } = (0, super_easy_validator_1.validate)(rules, Object.assign(Object.assign({}, req.body), req.params));
        if (errors) {
            return res.status(400).json({ message: errors[0] });
        }
        return next();
    }
    catch (error) {
        console.log(error);
        return res.status(500).json({ message: 'server error' });
    }
}
function putChat(req, res, next) {
    try {
        const rules = {
            id: 'mongoid',
            name: 'optional|string|min:3|max:30',
            restricted: 'optional|string|boolean',
            allowPublicPost: 'optional|string|boolean',
            category: 'optional|string|min:1|max:20'
        };
        const { errors } = (0, super_easy_validator_1.validate)(rules, Object.assign(Object.assign({}, req.params), req.body));
        if (errors) {
            return res.status(400).json({ message: errors[0] });
        }
        if (req.body.name) {
            req.body.name = req.body.name.trim();
            const { errors } = (0, super_easy_validator_1.validate)({ name: 'string|min:3|max:25' }, req.body);
            if (errors) {
                return res.status(400).json({ message: errors[0] });
            }
        }
        return next();
    }
    catch (error) {
        console.log(error);
        return res.status(500).json({ message: 'server error' });
    }
}
function postChatParticipantStatus(req, res, next) {
    try {
        const rules = {
            id: 'mongoid',
            participantId: 'mongoid',
            status: 'string|enums:accepted,rejected'
        };
        const { errors } = (0, super_easy_validator_1.validate)(rules, Object.assign(Object.assign({}, req.params), req.body));
        if (errors) {
            return res.status(400).json({ message: errors[0] });
        }
        return next();
    }
    catch (error) {
        console.log(error);
        return res.status(500).json({ message: 'server error' });
    }
}
function validateChatId(req, res, next) {
    try {
        const rules = {
            id: 'mongoid'
        };
        const { errors } = (0, super_easy_validator_1.validate)(rules, req.params);
        if (errors) {
            return res.status(400).json({ message: errors[0] });
        }
        return next();
    }
    catch (error) {
        console.log(error);
        return res.status(500).json({ message: 'server error' });
    }
}
function deleteChatParticipant(req, res, next) {
    try {
        const rules = {
            id: 'mongoid',
            participantId: 'mongoid',
            socket: `optional|${constants_1.default.socketRegex}`
        };
        const { errors } = (0, super_easy_validator_1.validate)(rules, Object.assign(Object.assign({}, req.params), req.body));
        if (errors) {
            return res.status(400).json({ message: errors[0] });
        }
        return next();
    }
    catch (error) {
        console.log(error);
        return res.status(500).json({ message: 'server error' });
    }
}
function validateChatParticipantId(req, res, next) {
    try {
        const rules = {
            id: 'mongoid',
            participantId: 'mongoid',
        };
        const { errors } = (0, super_easy_validator_1.validate)(rules, req.params);
        if (errors) {
            return res.status(400).json({ message: errors[0] });
        }
        return next();
    }
    catch (error) {
        console.log(error);
        return res.status(500).json({ message: 'server error' });
    }
}
const validations = {
    getChats,
    getExploreChats,
    getChatFiles,
    postChat,
    postChatParticipants,
    postChatParticipant,
    putChat,
    postChatParticipantStatus,
    validateChatId,
    validateChatParticipantId,
    deleteChatParticipant
};
exports.default = validations;
