"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const super_easy_validator_1 = require("super-easy-validator");
const startTimeRegex = /^(0?[1-9]|1[0-2]):[0-5][0-9] (AM|PM)$/;
function getEvents(req, res, next) {
    try {
        const rules = {
            limit: 'optional|string|natural',
            page: 'optional|string|natural',
            search: 'optional|string',
            type: 'optional|string|enums:others,mine',
            categories: 'optional|string|min:1'
        };
        const { errors } = (0, super_easy_validator_1.validate)(rules, req.query);
        if (errors) {
            return res.status(400).json({ message: errors[0] });
        }
        return next();
    }
    catch (error) {
        console.log(error);
        return res.status(500).json({ message: 'server error' });
    }
}
function checkEventId(req, res, next) {
    try {
        const rules = {
            id: 'mongoid'
        };
        const { errors } = (0, super_easy_validator_1.validate)(rules, req.params);
        if (errors) {
            return res.status(400).json({ message: errors[0] });
        }
        return next();
    }
    catch (error) {
        console.log(error);
        return res.status(500).json({ message: 'server error' });
    }
}
function postEvent(req, res, next) {
    try {
        const rules = {
            title: 'string|min:5|max:200',
            description: 'string|min:10|max:5000',
            date: 'dateonly',
            time: 'string',
            locationString: 'string|min:3|max:100',
            latitude: 'string|number|min:-90|max:90',
            longitude: 'string|number|min:-180|max:180',
            category: 'string|min:2|max:20',
        };
        const { errors } = (0, super_easy_validator_1.validate)(rules, req.body);
        if (errors) {
            return res.status(400).json({ message: errors[0] });
        }
        if (!startTimeRegex.test(req.body.time)) {
            return res.status(400).json({ message: 'time is invalid' });
        }
        return next();
    }
    catch (error) {
        console.log(error);
        return res.status(500).json({ message: 'server error' });
    }
}
function patchEvent(req, res, next) {
    try {
        const rules = {
            id: 'mongoid',
            title: 'optional|string|min:5|max:200',
            description: 'optional|string|min:10|max:5000',
            date: 'optional|dateonly',
            time: 'optional|string',
            locationString: 'optional|string|min:3|max:100',
            latitude: 'optional|string|number|min:-90|max:90',
            longitude: 'optional|string|number|min:-180|max:180',
            category: 'optional|string|min:2|max:20',
        };
        const { errors } = (0, super_easy_validator_1.validate)(rules, Object.assign(Object.assign({}, req.body), req.params));
        if (errors) {
            return res.status(400).json({ message: errors[0] });
        }
        const { date, time, latitude, longitude } = req.body;
        if ((latitude && !longitude) || (longitude && !latitude)) {
            return res.status(400).json({ message: 'both latitude & longitude are required' });
        }
        if ((date && !time) || (time && !date)) {
            return res.status(400).json({ message: 'both date & time are required' });
        }
        if (time && !startTimeRegex.test(time)) {
            return res.status(400).json({ message: 'time is invalid' });
        }
        return next();
    }
    catch (error) {
        console.log(error);
        return res.status(500).json({ message: 'server error' });
    }
}
const validations = {
    getEvents,
    checkEventId,
    postEvent,
    patchEvent
};
exports.default = validations;
