"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const super_easy_validator_1 = require("super-easy-validator");
const services_1 = __importDefault(require("../../utils/services"));
const constants_1 = __importDefault(require("../../utils/constants"));
function getReels(req, res, next) {
    try {
        const rules = {
            limit: 'optional|string|natural',
            dynamic: 'optional|string|boolean',
            chips: 'optional|string|enums:focus,mindfulness,reflection',
        };
        const { errors } = (0, super_easy_validator_1.validate)(rules, req.query);
        if (errors) {
            return res.status(400).json({ message: errors[0] });
        }
        return next();
    }
    catch (error) {
        console.log(error);
        return res.status(500).json({ message: 'server error' });
    }
}
function getExploreReels(req, res, next) {
    try {
        const rules = {
            limit: 'optional|string|natural',
            page: 'optional|string|natural',
            chat: 'optional|mongoid',
            tag: 'optional|string|min:1|max:20',
        };
        const { errors } = (0, super_easy_validator_1.validate)(rules, req.query);
        if (errors) {
            return res.status(400).json({ message: errors[0] });
        }
        return next();
    }
    catch (error) {
        console.log(error);
        return res.status(500).json({ message: 'server error' });
    }
}
function getUserReels(req, res, next) {
    try {
        const rules = {
            limit: 'optional|string|natural',
            page: 'optional|string|natural',
            userId: 'optional|mongoid'
        };
        const { errors } = (0, super_easy_validator_1.validate)(rules, Object.assign(Object.assign({}, req.query), req.params));
        if (errors) {
            return res.status(400).json({ message: errors[0] });
        }
        return next();
    }
    catch (error) {
        console.log(error);
        return res.status(500).json({ message: 'server error' });
    }
}
function checkReelId(req, res, next) {
    try {
        const { errors } = (0, super_easy_validator_1.validate)({ id: 'mongoid' }, req.params);
        if (errors) {
            return res.status(400).json({ message: errors[0] });
        }
        return next();
    }
    catch (error) {
        console.log(error);
        return res.status(500).json({ message: 'server error' });
    }
}
function postPresignUrl(req, res, next) {
    try {
        const rules = {
            id: 'optional|mongoid',
            files: 'arrayof:string|arrayof:min:5|arrayof:max:300'
        };
        const { errors } = (0, super_easy_validator_1.validate)(rules, req.body);
        if (errors) {
            return res.status(400).json({ message: errors[0] });
        }
        return next();
    }
    catch (error) {
        console.log(error);
        return res.status(500).json({ message: 'server error' });
    }
}
function postReel(req, res, next) {
    try {
        const rules = {
            id: 'mongoid',
            description: 'optional|string|min:3|max:200',
            tag: 'optional|string|min:1|max:30',
            subTag: 'optional|string|min:1|max:50',
            video: 'optional|url',
            chat: 'optional|mongoid',
            cover: 'optional|url',
            photos: 'optional|array|arrayof:url',
            dynamic: 'optional|object',
            $atleast: 'video|photos|dynamic',
            $atmost: 'video|photos|dynamic',
            hashtags: 'optional|array|arrayof:string|arrayof:min:1|arrayof|max:35',
            isAnonymous: 'optional|boolean',
            socket: `optional|${constants_1.default.socketRegex}`
        };
        const { errors } = (0, super_easy_validator_1.validate)(rules, req.body);
        if (errors) {
            return res.status(400).json({ message: errors[0] });
        }
        if (req.body.dynamic) {
            try {
                req.body.dynamic = services_1.default.nestedDynamicToFlat(req.body.dynamic);
                const rules = {
                    dynamic: [
                        {
                            _id: 'string|min:5|max:100',
                            label: 'optional|string|min:1|max:200',
                            parent: 'optional|string|min:5|max:100',
                            cover: 'optional|url',
                            video: 'url',
                            question: 'optional|string|min:3|max:200'
                        }
                    ]
                };
                const { errors } = (0, super_easy_validator_1.validate)(rules, req.body);
                if (errors) {
                    return res.status(400).json({ message: errors[0] });
                }
            }
            catch (error) {
                return res.status(400).json({ message: 'unable to parse dynamic' });
            }
        }
        return next();
    }
    catch (error) {
        console.log(error);
        return res.status(500).json({ message: 'server error' });
    }
}
function patchReel(req, res, next) {
    try {
        const rules = {
            id: 'mongoid',
            tag: 'optional|string|min:1|max:30',
            subTag: 'optional|string|min:1|max:50',
            description: 'optional|string|min:3|max:200',
            cover: 'optional|url',
            hashtags: 'optional|array|arrayof:string|arrayof:min:1|arrayof|max:35',
            isAnonymous: 'optional|boolean'
        };
        const { errors } = (0, super_easy_validator_1.validate)(rules, Object.assign(Object.assign({}, req.body), req.params));
        if (errors) {
            return res.status(400).json({ message: errors[0] });
        }
        return next();
    }
    catch (error) {
        console.log(error);
        return res.status(500).json({ message: 'server error' });
    }
}
const validations = {
    getReels,
    getExploreReels,
    getUserReels,
    checkReelId,
    postPresignUrl,
    postReel,
    patchReel
};
exports.default = validations;
