"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const models_1 = require("../users/models");
const models_2 = require("./models");
const models_3 = require("../notifications/models");
const models_4 = require("../chats/models");
const services_1 = __importDefault(require("../../utils/services"));
function getRequests(req, res) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            const user = res.locals.user;
            let { limit = '10', page = '1', status } = req.query;
            limit = +limit;
            page = +page;
            const skip = (page - 1) * limit;
            const query = Object.assign({ user: user._id }, (status ? { status } : {}));
            let requests = yield models_2.Request.find(query)
                .sort({ createdAt: -1 })
                .skip(skip)
                .limit(limit)
                .populate({
                path: 'user',
                select: { name: 1, username: 1, bio: 1, photo: 1 }
            })
                .populate('chat');
            const count = yield models_2.Request.countDocuments(query);
            // @ts-ignore
            requests = requests.map(e => e.toJSON());
            for (const request of requests) {
                const chat = request.chat;
                const p = chat.participants.find(e => `${e.user}` === `${user._id}`);
                // @ts-ignore
                chat.areYouAdmin = !!(p === null || p === void 0 ? void 0 : p.isAdmin);
            }
            return res.json({ requests, count });
        }
        catch (error) {
            console.error(error);
            return res.status(500).json({ message: 'server error' });
        }
    });
}
function getRequest(req, res) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            const user = res.locals.user;
            const id = req.params.id;
            let request = yield models_2.Request.findOne({ _id: id, user: user._id })
                .populate({
                path: 'user',
                select: { name: 1, username: 1, bio: 1, photo: 1 }
            })
                .populate('chat');
            if (!request) {
                return res.status(404).json({ message: 'request not found' });
            }
            // @ts-ignore
            request = request.toJSON();
            const chat = request.chat;
            const p = chat.participants.find(e => `${e.user}` === `${user._id}`);
            // @ts-ignore
            chat.areYouAdmin = !!(p === null || p === void 0 ? void 0 : p.isAdmin);
            return res.json({ request });
        }
        catch (error) {
            console.error(error);
            return res.status(500).json({ message: 'server error' });
        }
    });
}
function postRequest(req, res) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            const user = res.locals.user;
            const chatId = req.body.chat;
            const chat = yield models_4.Chat.findById(chatId);
            if (!chat) {
                return res.status(404).json({ message: 'chat not found' });
            }
            if (!chat.restricted) {
                return res.status(403).json({ message: 'only allowed for restricted chat' });
            }
            if (!chat.isGroup) {
                return res.status(403).json({ message: 'only allowed for group chats' });
            }
            const participant = chat.participants.find(e => `${e.user}` === `${user._id}`);
            if (participant) {
                return res.status(409).json({ message: 'you already exist in this chat as a participant' });
            }
            let request = yield models_2.Request.findOne({
                user: user._id,
                status: 'pending',
                chat: chat._id
            });
            if (request) {
                return res.status(409).json({ message: 'a pending request for this chat already exist' });
            }
            request = yield models_2.Request.create({
                user: user._id,
                chat: chat._id,
            });
            (() => __awaiter(this, void 0, void 0, function* () {
                try {
                    const adminIds = chat.participants.filter(e => e.isAdmin).map(e => `${e.user}`);
                    const admins = yield models_1.User.find({ _id: { $in: adminIds } });
                    for (const adminId of adminIds) {
                        const admin = admins.find(e => `${e._id}` === adminId);
                        if (!admin) {
                            continue;
                        }
                        yield services_1.default.notifyUser({
                            title: 'A user wants to join the chat',
                            description: `${user.username} wants to join ${chat.name || 'your chat'}`,
                            event: 'chat-join-request',
                            user: admin,
                            from: user._id,
                            chat: chat._id,
                            request: request._id
                        });
                    }
                }
                catch (error) {
                    console.log(error);
                }
            }))();
            return res.status(201).json({ request });
        }
        catch (error) {
            console.log(error);
            return res.status(500).json({ message: 'server error' });
        }
    });
}
function patchRequest(req, res) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            const user = res.locals.user;
            const id = req.params.id;
            const status = req.body.status;
            const request = yield models_2.Request.findById(id).populate('chat').populate('user');
            if (!request) {
                return res.status(404).json({ message: 'request not found' });
            }
            const chat = request.chat;
            const participant = chat.participants.find(e => `${e.user}` === `${user._id}`);
            if (!participant) {
                return res.status(404).json({ message: 'request not found' });
            }
            if (!participant.isAdmin) {
                return res.status(403).json({ message: 'admin permission is required' });
            }
            if (request.status !== 'pending') {
                return res.status(403).json({ message: 'Request status must be pending' });
            }
            request.status = status;
            if (status === 'accepted') {
                request.acceptedAt = new Date();
            }
            else {
                request.rejectedAt = new Date();
            }
            yield request.save();
            if (status === 'accepted') {
                chat.participants.push({ user: request.user._id });
                yield chat.save();
            }
            ;
            (() => __awaiter(this, void 0, void 0, function* () {
                try {
                    // mark the notification as dismissed
                    const notification = yield models_3.Notification.findOne({
                        request: request._id,
                        event: 'chat-join-request'
                    });
                    if (!notification) {
                        return;
                    }
                    notification.dismissed = true;
                    yield notification.save();
                    // notify the user, in case of accepted
                    if (status === 'accepted') {
                        yield services_1.default.notifyUser({
                            title: 'Your join request has been accepted',
                            description: `You are now a member of ${chat.name || 'the chat'}`,
                            event: 'chat-request-accepted',
                            user: request.user,
                            from: user._id,
                            chat: chat._id,
                            request: request._id
                        });
                    }
                }
                catch (error) {
                    console.log(error);
                }
            }))();
            (() => __awaiter(this, void 0, void 0, function* () {
                try {
                    if (status !== 'accepted') {
                        return;
                    }
                    const score = yield services_1.default.getOrCreateTodayScore(request.user._id);
                    if (!score) {
                        return;
                    }
                    const action = score.social.actions.find(e => e.type === 'join-group-chat');
                    if (!action || action.completed) {
                        return;
                    }
                    action.value = 1;
                    action.completed = true;
                    yield services_1.default.updateScore(score);
                }
                catch (error) {
                    console.log(error);
                }
            }))();
            return res.json({ message: `request ${status} successfully` });
        }
        catch (error) {
            console.log(error);
            return res.status(500).json({ message: 'server error' });
        }
    });
}
const controllers = {
    getRequests,
    getRequest,
    postRequest,
    patchRequest
};
exports.default = controllers;
