"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const models_1 = require("../users/models");
const models_2 = require("./models");
const models_3 = require("../events/models");
const constants_1 = __importDefault(require("../../utils/constants"));
const services_1 = __importDefault(require("../../utils/services"));
function getStatuses(req, res) {
    return __awaiter(this, void 0, void 0, function* () {
        var _a, _b, _c;
        try {
            const user = res.locals.user;
            let { limit = '10', page = '1', search, categories } = req.query;
            limit = +limit;
            page = +page;
            const matchStage = {
                _id: { $ne: user._id },
                name: { $exists: true },
                status: { $exists: true },
                $or: [
                    { terminated: false },
                    { terminated: { $exists: false } }
                ]
            };
            if (search) {
                matchStage.name = { $regex: search, $options: 'i' };
            }
            const categoryFilter = categories
                ? { 'status.categories': { $in: categories.split(',') } }
                : {};
            const now = new Date();
            const afterUnwindMatch = Object.assign(Object.assign({}, categoryFilter), { 'status.visibleAt': { $lte: now }, 'status.endedAt': { $gte: now } });
            if (constants_1.default.locationMode && user.location) {
                afterUnwindMatch.$or = [
                    {
                        'status.location': {
                            $geoWithin: {
                                $centerSphere: [
                                    user.location.coordinates, // [lng, lat]
                                    constants_1.default.locationDistance / constants_1.default.earthRadius // meters → radians
                                ]
                            }
                        }
                    },
                    { 'status.location': { $exists: false } }
                ];
            }
            const pipeline = [
                { $match: matchStage },
                {
                    $lookup: {
                        from: 'statuses',
                        localField: 'status',
                        foreignField: '_id',
                        as: 'status'
                    }
                },
                { $unwind: { path: '$status', preserveNullAndEmptyArrays: false } },
                { $match: afterUnwindMatch },
                {
                    $lookup: {
                        from: 'connections',
                        let: { otherUserId: '$_id' },
                        pipeline: [
                            {
                                $match: {
                                    $expr: {
                                        $or: [
                                            { $and: [{ $eq: ['$from', user._id] }, { $eq: ['$to', '$$otherUserId'] }] },
                                            { $and: [{ $eq: ['$to', user._id] }, { $eq: ['$from', '$$otherUserId'] }] }
                                        ]
                                    }
                                }
                            },
                            { $limit: 1 }
                        ],
                        as: 'connection'
                    }
                },
                { $unwind: { path: '$connection', preserveNullAndEmptyArrays: true } },
                {
                    $project: {
                        name: 1,
                        username: 1,
                        bio: 1,
                        auth: {
                            emailVerified: 1,
                            phoneVerified: 1
                        },
                        status: 1,
                        connection: 1,
                        photo: 1,
                        createdAt: 1,
                        updatedAt: 1
                    }
                },
                { $sort: { createdAt: -1 } },
                {
                    $facet: {
                        users: [
                            { $skip: (page - 1) * limit },
                            { $limit: limit }
                        ],
                        totalCount: [
                            { $count: 'count' }
                        ]
                    }
                }
            ];
            const result = yield models_1.User.aggregate(pipeline);
            const users = ((_a = result[0]) === null || _a === void 0 ? void 0 : _a.users) || [];
            const count = ((_c = (_b = result[0]) === null || _b === void 0 ? void 0 : _b.totalCount[0]) === null || _c === void 0 ? void 0 : _c.count) || 0;
            (() => __awaiter(this, void 0, void 0, function* () {
                try {
                    if (!categories) {
                        return;
                    }
                    const score = yield services_1.default.getOrCreateTodayScore(user._id);
                    if (!score) {
                        return;
                    }
                    const action = score.social.actions.find(e => e.type === 'search-with-meetme-category-filter');
                    if (!action || action.completed) {
                        return;
                    }
                    action.value = 1;
                    action.completed = true;
                    yield services_1.default.updateScore(score);
                }
                catch (error) {
                    console.log(error);
                }
            }))();
            return res.json({ users, count });
        }
        catch (error) {
            console.error(error);
            return res.status(500).json({ message: 'server error' });
        }
    });
}
// meet me search
function getStatusesAndEvents(req, res) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            const user = res.locals.user;
            let { limit = '5', search = '' } = req.query;
            limit = +limit;
            // fetching users with their status
            const matchStage = {
                _id: { $ne: user._id },
                name: { $exists: true },
                status: { $exists: true },
                $or: [
                    { terminated: false },
                    { terminated: { $exists: false } }
                ]
            };
            const now = new Date();
            const afterUnwindMatch = {
                'status.visibleAt': { $lte: now },
                'status.endedAt': { $gte: now },
                $and: [
                    // search
                    {
                        $or: [
                            { name: { $regex: search, $options: 'i' } },
                            { 'status.title': { $regex: search, $options: 'i' } },
                        ]
                    },
                ]
            };
            if (constants_1.default.locationMode && user.location) {
                afterUnwindMatch.$and.push({
                    // location
                    $or: [
                        {
                            'status.location': {
                                $geoWithin: {
                                    $centerSphere: [
                                        user.location.coordinates, // [lng, lat]
                                        constants_1.default.locationDistance / constants_1.default.earthRadius // meters → radians
                                    ]
                                }
                            }
                        },
                        { 'status.location': { $exists: false } }
                    ]
                });
            }
            let pipeline = [
                { $match: matchStage },
                {
                    $lookup: {
                        from: 'statuses',
                        localField: 'status',
                        foreignField: '_id',
                        as: 'status'
                    }
                },
                { $unwind: { path: '$status', preserveNullAndEmptyArrays: false } },
                { $match: afterUnwindMatch },
                {
                    $lookup: {
                        from: 'connections',
                        let: { otherUserId: '$_id' },
                        pipeline: [
                            {
                                $match: {
                                    $expr: {
                                        $or: [
                                            { $and: [{ $eq: ['$from', user._id] }, { $eq: ['$to', '$$otherUserId'] }] },
                                            { $and: [{ $eq: ['$to', user._id] }, { $eq: ['$from', '$$otherUserId'] }] }
                                        ]
                                    }
                                }
                            },
                            { $limit: 1 }
                        ],
                        as: 'connection'
                    }
                },
                { $unwind: { path: '$connection', preserveNullAndEmptyArrays: true } },
                {
                    $project: {
                        name: 1,
                        username: 1,
                        bio: 1,
                        auth: {
                            emailVerified: 1,
                            phoneVerified: 1
                        },
                        status: 1,
                        connection: 1,
                        photo: 1,
                        createdAt: 1,
                        updatedAt: 1
                    }
                },
                { $sort: { createdAt: -1 } },
                { $limit: limit },
            ];
            const users = yield models_1.User.aggregate(pipeline);
            // fetching events
            const match = {
                endedAt: { $gt: new Date() },
                $or: [
                    { title: { $regex: search, $options: 'i' } },
                    { description: { $regex: search, $options: 'i' } },
                ]
            };
            if (constants_1.default.locationMode && user.location) {
                match.location = {
                    $geoWithin: {
                        $centerSphere: [
                            user.location.coordinates, // [lng, lat]
                            constants_1.default.locationDistance / constants_1.default.earthRadius // meters → radians
                        ]
                    }
                };
            }
            // Main aggregation
            pipeline = [
                { $match: match },
                { $sort: { createdAt: -1 } },
                {
                    $lookup: {
                        from: 'chats',
                        localField: 'chat',
                        foreignField: '_id',
                        as: 'chat',
                    },
                },
                { $unwind: '$chat' },
                {
                    $addFields: {
                        participantIds: '$chat.participants.user',
                        'chat.participantsCount': { $size: '$chat.participants' }
                    },
                },
                {
                    $match: {
                        participantIds: { $ne: user._id },
                    },
                },
                { $limit: limit }
            ];
            const events = yield models_3.Event.aggregate(pipeline);
            for (const event of events) {
                delete event.participantIds;
                event.joined = event.chat.participants.some((e) => `${e.user}` === `${user._id}`);
                const p = event.chat.participants.find((e) => `${e.user}` === `${user._id}`);
                event.chat.isJoined = !!p;
                event.chat.areYouAdmin = !!(p === null || p === void 0 ? void 0 : p.isAdmin);
            }
            return res.json({ users, events });
        }
        catch (error) {
            console.log(error);
            return res.status(500).json({ message: 'server error' });
        }
    });
}
function postStatus(req, res) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            const user = res.locals.user;
            const { categories, title = '', schedule, startDateType, startDate, startTime, latitude, longitude, locationString } = req.body;
            let _status = yield models_2.Status.findOne({ user: user._id });
            if (_status && _status.endedAt > new Date()) {
                return res.status(409).json({ message: 'status already exist' });
            }
            if (_status) {
                user.status = undefined;
                yield _status.deleteOne();
            }
            let location = user.location;
            if (typeof latitude === 'number' && typeof longitude === 'number') {
                location = {
                    type: 'Point',
                    coordinates: [longitude, latitude]
                };
            }
            const { startedAt, visibleAt, endedAt } = getStartedEndedStatusTime({ startDate, startTime });
            const status = yield models_2.Status.create({
                user: user._id,
                categories,
                title,
                schedule,
                startDateType,
                startDate,
                startTime,
                startedAt,
                visibleAt,
                endedAt,
                locationString: locationString !== null && locationString !== void 0 ? locationString : user.locationString,
                location,
            });
            user.status = status._id;
            yield user.save();
            (() => __awaiter(this, void 0, void 0, function* () {
                try {
                    const score = yield services_1.default.getOrCreateTodayScore(user._id);
                    if (!score) {
                        return;
                    }
                    const action = score.social.actions.find(e => e.type === 'create-meetme-status');
                    if (!action || action.completed) {
                        return;
                    }
                    action.value = 1;
                    action.completed = true;
                    yield services_1.default.updateScore(score);
                }
                catch (error) {
                    console.log(error);
                }
            }))();
            return res.status(201).json({ status });
        }
        catch (error) {
            console.log(error);
            return res.status(500).json({ message: 'server error' });
        }
    });
}
function patchStatus(req, res) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            const user = res.locals.user;
            const { categories, title, schedule, startDateType, startDate, startTime, latitude, longitude, locationString } = req.body;
            const status = yield models_2.Status.findOne({ user: user._id });
            if (!status) {
                return res.status(404).json({ message: 'status not found' });
            }
            if (startDate || startTime) {
                const { startedAt, visibleAt, endedAt } = getStartedEndedStatusTime({ startDate: startDate !== null && startDate !== void 0 ? startDate : status.startDate, startTime: startTime !== null && startTime !== void 0 ? startTime : status.startTime });
                status.startedAt = startedAt;
                status.visibleAt = visibleAt;
                status.endedAt = endedAt;
            }
            status.categories = categories !== null && categories !== void 0 ? categories : status.categories;
            if (title !== undefined) {
                status.title = title;
            }
            status.schedule = schedule !== null && schedule !== void 0 ? schedule : status.schedule;
            status.startDateType = startDateType !== null && startDateType !== void 0 ? startDateType : status.startDateType;
            status.locationString = locationString !== null && locationString !== void 0 ? locationString : status.locationString;
            if (typeof latitude === 'number' && typeof longitude === 'number') {
                status.location = {
                    type: 'Point',
                    coordinates: [longitude, latitude]
                };
            }
            yield status.save();
            return res.json({ status });
        }
        catch (error) {
            console.log(error);
            return res.status(500).json({ message: 'server error' });
        }
    });
}
function deleteStatus(req, res) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            const user = res.locals.user;
            const status = yield models_2.Status.findOne({ user: user._id });
            if (!status) {
                return res.status(404).json({ message: 'status not found' });
            }
            yield status.deleteOne();
            user.status = undefined;
            yield user.save();
            return res.json({ message: 'status deleted successfully' });
        }
        catch (error) {
            console.log(error);
            return res.status(500).json({ message: 'server error' });
        }
    });
}
function getStartedEndedStatusTime(data) {
    const { startDate, startTime } = data;
    const startedAt = new Date(`${startDate} ${startTime}`);
    // const visibleAt = new Date(startedAt.getTime() - 12 * 60 * 60 * 1000) // 12 hours earlier
    const visibleAt = new Date(startedAt); // same as startedAt
    const endedAt = new Date(startedAt.getTime() + 3 * 60 * 60 * 1000); // 3 hour late
    return { startedAt, visibleAt, endedAt };
}
const controllers = {
    getStatuses,
    getStatusesAndEvents,
    postStatus,
    patchStatus,
    deleteStatus
};
exports.default = controllers;
