"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const express_rate_limit_1 = __importDefault(require("express-rate-limit"));
const cors_1 = __importDefault(require("cors"));
const helmet_1 = __importDefault(require("helmet"));
const multer_1 = require("multer");
const mongoose_1 = require("mongoose");
const client_s3_1 = require("@aws-sdk/client-s3");
const openai_1 = __importDefault(require("openai"));
const firebase_admin_1 = __importDefault(require("firebase-admin"));
const firebase_json_1 = __importDefault(require("../../firebase.json"));
const p_queue_1 = __importDefault(require("p-queue"));
const client_scheduler_1 = require("@aws-sdk/client-scheduler");
const schedulerClient = new client_scheduler_1.SchedulerClient({
    region: process.env.AWS_REGION,
    credentials: {
        accessKeyId: process.env.AWS_ACCESS_KEY_ID,
        secretAccessKey: process.env.AWS_SECRET_ACCESS_KEY,
    }
});
const limiter = (0, express_rate_limit_1.default)({
    windowMs: 1 * 60 * 1000, // 1 minutes
    max: 80,
    handler: (_req, res) => res.status(429).json({ message: 'Too many requests' }),
});
// environment
const environment = process.env.NODE_ENV;
const isProduction = environment === 'production';
const isDevelopment = environment === 'development';
// cors
const corsValue = isProduction ? {
    origin: [
        process.env.FRONTEND_HOST,
        process.env.BACKEND_HOST
    ]
} : undefined;
const cors = (0, cors_1.default)(corsValue);
// helmet
const helmet = (0, helmet_1.default)({ contentSecurityPolicy: false });
// error handler
const errorHandler = (error, _req, res, _next) => {
    console.log(error);
    if (!(error instanceof multer_1.MulterError)) {
        return res.status(500).json({ message: 'server error' });
    }
    if (error.code === 'LIMIT_FILE_SIZE') {
        return res.status(413).json({ message: 'file size too large' });
    }
    return res.status(400).json({ message: error.message });
};
// mongoose connection
const mongooseConnection = (0, mongoose_1.connect)(process.env.MONGO_URL, {
    dbName: process.env.DATABASE,
});
// firebase admin
firebase_admin_1.default.initializeApp({
    credential: firebase_admin_1.default.credential.cert(firebase_json_1.default),
});
const R2 = new client_s3_1.S3Client({
    endpoint: process.env.R2_ENDPOINT,
    region: process.env.R2_REGION,
    credentials: {
        accessKeyId: process.env.R2_ACCESS_KEY_ID,
        secretAccessKey: process.env.R2_SECRET_ACCESS_KEY,
    },
});
const key = Buffer.from(process.env.AES_KEY);
const iv = Buffer.from(process.env.AES_IV);
const openai = new openai_1.default({ apiKey: process.env.OPENAI_API_KEY });
const compressionQueue = new p_queue_1.default({
    concurrency: +(process.env.MAX_PARALLEL_COMPRESSION || '3')
});
const config = {
    limiter,
    cors,
    helmet,
    isDevelopment,
    isProduction,
    errorHandler,
    mongooseConnection,
    firebaseAdmin: firebase_admin_1.default,
    R2,
    key,
    iv,
    openai,
    compressionQueue,
    schedulerClient
};
exports.default = config;
