"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const jsonwebtoken_1 = __importDefault(require("jsonwebtoken"));
const models_1 = require("../api/users/models");
const constants_1 = __importDefault(require("./constants"));
const services_1 = __importDefault(require("./services"));
const helpers_1 = __importDefault(require("./helpers"));
function authorize(socket, next) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            const token = socket.handshake.auth.token;
            let payload;
            try {
                payload = jsonwebtoken_1.default.verify(token, process.env.JWT_SECRET);
            }
            catch (error) {
                socket.data.error = { event: 'authorization-failed', data: { message: 'authorization failed' } };
                return next();
            }
            const { _id, hash } = payload;
            const user = yield models_1.User.findById(_id);
            if (!user) {
                socket.data.error = { event: 'authorization-failed', data: { message: 'authorization failed' } };
                return next();
            }
            if (user.auth.password.slice(-10) !== hash) {
                socket.data.error = { event: 'authorization-failed', data: { message: 'authorization failed' } };
                return next();
            }
            const tokenHash = helpers_1.default.getHash(token);
            const device = user.auth.loggedinDevices.find(e => e.tokenHash === tokenHash);
            if (!device) {
                socket.data.error = { event: 'authorization-failed', data: { message: 'authorization failed' } };
                return next();
            }
            if (!user.auth.emailVerified) {
                socket.data.error = { event: 'verification-required', data: { message: 'Please verify your email first' } };
                return next();
            }
            if (user.terminated) {
                socket.data.error = { event: 'account-terminated', data: { message: 'Your account has been terminated due to a violation of our Terms and Conditions and in accordance with our Privacy Policy' } };
                return next();
            }
            socket.data.user = user;
            socket.data.device = device;
            return next();
        }
        catch (error) {
            console.error(error);
            socket.data.error = { event: 'server-error', data: { message: 'Server error' } };
            return next();
        }
    });
}
function onInit(socket) {
    return __awaiter(this, void 0, void 0, function* () {
        var _a, _b, _c;
        var _d, _e;
        try {
            if (socket.data.error) {
                socket.emit('error', socket.data.error);
                console.log(`❌ socket error message sent`);
                console.dir(socket.data.error, { depth: null });
                return socket.disconnect(true);
            }
            const user = socket.data.user;
            console.log(`✅ User connected: ${socket.id} (${(_b = (_a = user.username) !== null && _a !== void 0 ? _a : user.email) !== null && _b !== void 0 ? _b : user._id})`);
            (_c = (_d = constants_1.default.sockets)[_e = `${user._id}`]) !== null && _c !== void 0 ? _c : (_d[_e] = []);
            constants_1.default.sockets[`${user._id}`].push(socket);
            socket.on('disconnect', () => onDisconnect(socket));
            socket.on('error', (error) => console.error(error));
            socket.onAny((event, data) => onMessage(socket, data));
            if (!user.isManuallyOffline) {
                services_1.default.socketNotifyOnUserPresenceUpdated(user, 'online');
            }
            yield models_1.User.updateOne({ _id: user._id }, { $set: { isOnline: true, lastOnlineAt: new Date() } });
        }
        catch (error) {
            console.log(error);
        }
    });
}
function onMessage(socket, payload) {
    try {
        const { event, data } = payload;
        console.log(`💬 Socket received: ${event}`, payload);
        const handler = constants_1.default.socketHandlers[event];
        try {
            handler === null || handler === void 0 ? void 0 : handler(socket, data);
        }
        catch (err) {
            console.log(err);
            services_1.default.socketErrorNotifyUser(socket, 'server-error', { message: 'server error' });
        }
    }
    catch (error) {
        console.error(error);
    }
}
function onDisconnect(socket) {
    return __awaiter(this, void 0, void 0, function* () {
        var _a, _b;
        try {
            const user = socket.data.user;
            if (!user) {
                return console.log(`❌ User connection failed: ${socket.id}`);
            }
            let sockets = constants_1.default.sockets[`${user._id}`] || [];
            sockets = sockets.filter(e => e.id !== socket.id);
            constants_1.default.sockets[`${user._id}`] = sockets;
            if (!sockets.length) {
                delete constants_1.default.sockets[`${user._id}`];
            }
            console.log(`❌ User disconnected: ${socket.id} (${(_b = (_a = user.username) !== null && _a !== void 0 ? _a : user.email) !== null && _b !== void 0 ? _b : user._id})`);
            yield models_1.User.updateOne({ _id: user._id }, {
                $set: { lastOnlineAt: new Date() },
                $unset: { isOnline: '' }
            });
            const at = new Date();
            at.setMinutes(at.getMinutes() + constants_1.default.maxOnlineVisibilityInMinutes);
            yield services_1.default.createScheduler('user-offline', at, { user: `${user._id}` });
        }
        catch (error) {
            console.log(error);
        }
    });
}
const sockets = {
    authorize,
    onInit,
    onMessage,
    onDisconnect
};
exports.default = sockets;
