require('dotenv').config()

import express from 'express'
import morgan from 'morgan'
import cron from './src/utils/cron'
import config from './src/utils/config'
import constants from './src/utils/constants'
import colors from 'colors'
import services from './src/utils/services'
import http from 'http'
import { Server } from 'socket.io'
import sockets from './src/utils/sockets'
import userSockets from './src/api/users/sockets'
import messageSockets from './src/api/messages/sockets'
import chatSockets from './src/api/chats/sockets'
import userSchedules from './src/api/users/schedules'
import { ScheduleEvent, ScheduleEventHandler } from './src/utils/types'

const app = express()
app.use(config.helmet)
app.use(morgan('dev'))
app.use(express.json())
app.use('/docs', express.static('public'))
app.use(config.cors)
if (!config.isProduction) {
	app.use('/api', config.limiter)
}
app.all(['/', '/api'], (_, res) => res.json({ message: 'server running ⚡⚡' }))

// User API routes
app.use('/api/auth', require('./src/api/auth/routes'))
app.use('/api/users', require('./src/api/users/routes'))
app.use('/api/contacts', require('./src/api/contacts/routes'))
app.use('/api/followers', require('./src/api/followers/routes'))
app.use('/api/statuses', require('./src/api/statuses/routes'))
app.use('/api/connections', require('./src/api/connections/routes'))
app.use('/api/reels', require('./src/api/reels/routes'))
app.use('/api/views', require('./src/api/views/routes'))
app.use('/api/shares', require('./src/api/shares/routes'))
app.use('/api/likes', require('./src/api/likes/routes'))
app.use('/api/super-likes', require('./src/api/super-likes/routes'))
app.use('/api/comments', require('./src/api/comments/routes'))
app.use('/api/favorites', require('./src/api/favorites/routes'))
app.use('/api/blockings', require('./src/api/blockings/routes'))
app.use('/api/notifications', require('./src/api/notifications/routes'))
app.use('/api/chats', require('./src/api/chats/routes'))
app.use('/api/messages', require('./src/api/messages/routes'))
app.use('/api/requests', require('./src/api/requests/routes'))
app.use('/api/events', require('./src/api/events/routes'))
app.use('/api/contact-us', require('./src/api/contact-us/routes'))
app.use('/api/reports', require('./src/api/reports/routes'))
app.use('/api/rewards', require('./src/api/rewards/routes'))
app.use('/api/reward-hashes', require('./src/api/reward-hashes/routes'))
app.use('/api/scores', require('./src/api/scores/routes'))
app.use('/api/misc', require('./src/api/misc/routes'))
app.use('/api/usages', require('./src/api/usages/routes'))

// Admin API routes
app.use('/api/admin', require('./src/api/admin/routes'))
app.use('/api/admin/auth', require('./src/api/auth/admin/routes'))
app.use('/api/admin/users', require('./src/api/users/admin/routes'))
app.use('/api/admin/rewards', require('./src/api/rewards/admin/routes'))
app.use('/api/admin/reels', require('./src/api/reels/admin/routes'))

app.use(config.errorHandler)
app.use((req, res) => res.status(405).json({ message: 'route not implemented' }))

const server = http.createServer(app)
constants.io = new Server(server)

// deleting all files in temp folder
services.clearTempFolder()

server.listen(process.env.PORT, async () => {
	console.log('INFO: '.green + 'Server started!')

	// connecting mongodb
	const client = await config.mongooseConnection
	constants.db = client.connection
	console.log('INFO: '.green + 'Atlas connected!')
	console.log('INFO: '.green + 'Cronjob started!')

	// socket init
	constants.io.use(sockets.authorize)
	constants.io.on('connection', sockets.onInit)
	constants.socketHandlers = { ...userSockets, ...messageSockets, ...chatSockets }
	constants.scheduleHandlers = { ...userSchedules } as Record<ScheduleEvent, ScheduleEventHandler>
})

constants.firebaseAdmin = config.firebaseAdmin
constants.cron = cron
constants.colors = colors
services
