import { Request, Response, NextFunction } from 'express'
import { validate } from 'super-easy-validator'

function getComments(req: Request, res: Response, next: NextFunction) {
	try {
		const rules = {
			reel: 'mongoid',
			parent: 'optional|mongoid',
			limit: 'optional|string|natural',
			page: 'optional|string|natural',
		}
		const { errors } = validate(rules, req.query)
		if (errors) {
			return res.status(400).json({ message: errors[0] })
		}

		return next()
	} catch (error) {
		console.log(error)
		return res.status(500).json({ message: 'server error' })
	}
}

function postComment(req: Request, res: Response, next: NextFunction) {
	try {
		const rules = {
			reel: 'mongoid',
			parent: 'optional|mongoid',
			data: [
				{
					text: 'optional|string|min:1|max:500',
					user: 'optional|mongoid',
					$atleast: 'text|user'
				}
			]
		}
		const { errors } = validate(rules, req.body)
		if (errors) {
			return res.status(400).json({ message: errors[0] })
		}

		return next()
	} catch (error) {
		console.log(error)
		return res.status(500).json({ message: 'server error' })
	}
}

function patchComment(req: Request, res: Response, next: NextFunction) {
	try {
		const rules = {
			id: 'mongoid',
			data: [
				{
					text: 'optional|string|min:1|max:500',
					user: 'optional|mongoid',
					$atleast: 'text|user'
				}
			]
		}
		const { errors } = validate(rules, {...req.body, ...req.params})
		if (errors) {
			return res.status(400).json({ message: errors[0] })
		}

		return next()
	} catch (error) {
		console.log(error)
		return res.status(500).json({ message: 'server error' })
	}
}

function deleteComment(req: Request, res: Response, next: NextFunction) {
	try {
		const rules = {
			id: 'mongoid'
		}
		const { errors } = validate(rules, req.params)
		if (errors) {
			return res.status(400).json({ message: errors[0] })
		}

		return next()
	} catch (error) {
		console.log(error)
		return res.status(500).json({ message: 'server error' })
	}
}

const validations = {
	getComments,
	postComment,
	patchComment,
	deleteComment
}

export default validations
