import { Request, Response } from 'express'
import { IUser, User } from '../users/models'
import { Follower } from './models'
import services from '../../utils/services'
import { MutualFollow } from '../mutual-follows/models'

async function getFollowers(req: Request, res: Response) {
	interface Payload {
		limit?: string | number
		page?: string | number
		user?: string
		type?: 'followers' | 'followings'
	}

	try {
		const user = res.locals.user as IUser
		let { limit = '10', page = '1', user: userId, type = 'followers' } = req.query as unknown as Payload
		limit = +limit
		page = +page

		const targetUserId = userId || user._id.toString()

		if (userId) {
			const targetUser = await User.findById(userId)
			if (!targetUser) {
				return res.status(404).json({ message: 'user not found' })
			}
		}

		let match: any = {}
		if (type === 'followers') {
			// All users who follow targetUserId
			match = { following: targetUserId }
		} else {
			// All users targetUserId is following
			match = { follower: targetUserId }
		}

		const followersDocs = await Follower.find(match)
			.skip((page - 1) * limit)
			.limit(limit)

		// Extract the IDs of the other users we want details for
		const userIds = followersDocs.map(doc =>
			type === 'followers' ? doc.follower : doc.following
		)

		const users = await User.find(
			{ _id: { $in: userIds } },
			{
				name: 1,
				username: 1,
				bio: 1,
				auth: { emailVerified: 1, phoneVerified: 1 },
				photo: 1,
				createdAt: 1
			}
		)

		const count = await Follower.countDocuments(match)

		return res.json({ users, count })
	} catch (error) {
		console.error(error)
		return res.status(500).json({ message: 'server error' })
	}
}

async function postFollower(req: Request, res: Response) {
	try {
		const userId = req.params.userId as string
		const user = res.locals.user as IUser

		if (userId === user._id.toString()) {
			return res.status(403).json({ message: 'cannot follow yourself' })
		}

		let [targetUser, follower] = await Promise.all([
			User.findById(userId),
			Follower.findOne({ follower: user._id, following: userId })
		])
		if (!targetUser) {
			return res.status(404).json({ message: 'user not found' })
		}
		if (follower) {
			return res.json({ message: 'following successfully' })
		}

		const isBlocked = await services.isUserBlocked(user._id, targetUser._id)
		if(isBlocked) {
			return res.status(403).json({message: "Can't unfollow when its blocked"})
		}

		follower = await Follower.create({
			follower: user._id,
			following: userId
		});

		(async () => {
			try {
				const currentUserFollowingsCount = await Follower.countDocuments({ follower: user._id })
				user.followings = currentUserFollowingsCount
				await user.save()
				const targetUserFollowersCount = await Follower.countDocuments({ following: userId })
				await User.updateOne({_id: userId}, {$set: {followers: targetUserFollowersCount}})
				const following = await Follower.findOne({ following: user._id, follower: userId })
				if(following) {
					const mutualFollow = await MutualFollow.findOne({users: {$all: [user._id, targetUser._id]}})
					if(!mutualFollow) {
						await MutualFollow.create({users: [user._id, targetUser._id]})
					}
				}

				await services.notifyUser({
					title: 'Your New Follower',
					description: `${user.username} has started following you`,
					event: 'user-followed',
					user: targetUser as IUser,
					follower: follower._id,
					from: user._id,
				})
			} catch (error) {
				console.log(error)
			}
		})()

		return res.status(201).json({ message: 'following successfully' })
	} catch (error) {
		console.log(error)
		return res.status(500).json({ message: 'server error' })
	}
}

async function deleteFollower(req: Request, res: Response) {
	try {
		const userId = req.params.userId as string
		const user = res.locals.user as IUser

		if (userId === user._id.toString()) {
			return res.status(400).json({ message: 'Cannot unfollow yourself' })
		}

		const targetUser = await User.findById(userId)
		if (!targetUser) {
			return res.status(404).json({ message: 'user not found' })
		}
		const isBlocked = await services.isUserBlocked(user._id, targetUser._id)
		if(isBlocked) {
			return res.status(403).json({message: "Can't unfollow when its blocked"})
		}

		await Follower.deleteOne({ follower: user._id, following: userId });

		(async () => {
			try {
				const currentUserFollowingsCount = await Follower.countDocuments({ follower: user._id })
				user.followings = currentUserFollowingsCount
				await user.save()
				const targetUserFollowersCount = await Follower.countDocuments({ following: userId })
				targetUser.followers = targetUserFollowersCount
				await targetUser.save()
				await MutualFollow.deleteOne({users: {$all: [user._id, targetUser._id]}})
			} catch (error) {
				console.log(error)
			}
		})()

		return res.json({ message: 'unfollowed successfully' })
	} catch (error) {
		console.log(error)
		return res.status(500).json({ message: 'server error' })
	}
}

const controllers = {
	getFollowers,
	postFollower,
	deleteFollower
}

export default controllers
