import { Request, Response, NextFunction } from 'express'
import { validate } from 'super-easy-validator'
import constants from '../../utils/constants'

function getMessages(req: Request, res: Response, next: NextFunction) {
	try {
		const rules = {
			limit: 'optional|string|natural',
			page: 'optional|string|natural'
		}

		const { errors } = validate(rules, req.query)
		if (errors) {
			return res.status(400).json({ message: errors[0] })
		}

		return next()
	} catch (error) {
		console.log(error)
		return res.status(500).json({ message: 'server error' })
	}
}

function postMessage(req: Request, res: Response, next: NextFunction) {
	try {
		const rules = {
			chat: 'mongoid',
			content: 'optional|string|min:1|max:2000',
			files: 'optional|array|min:1|max:20|arrayof:object',
			socket: `optional|${constants.socketRegex}`
		}

		const { errors } = validate(rules, req.body)
		if (errors) {
			return res.status(400).json({ message: errors[0] })
		}

		if(req.body.files) {
			const rules = {
				files: [
					{
						url: 'url',
						cover: 'optional|url',
						meta: 'optional'
					}
				]
			}
			const { errors } = validate(rules, req.body)
			if (errors) {
				return res.status(400).json({ message: errors[0] })
			}
		} 

		return next()
	} catch (error) {
		console.log(error)
		return res.status(500).json({ message: 'server error' })
	}
}

function postMessageShareReels(req: Request, res: Response, next: NextFunction) {
	try {
		const rules = {
			chats: 'arrayof:mongoid',
			reel: 'mongoid',
			socket: `optional|${constants.socketRegex}`
		}

		const { errors } = validate(rules, req.body)
		if (errors) {
			return res.status(400).json({ message: errors[0] })
		}

		return next()
	} catch (error) {
		console.log(error)
		return res.status(500).json({ message: 'server error' })
	}
}

function patchMessage(req: Request, res: Response, next: NextFunction) {
	try {
		const rules = {
			id: 'mongoid',
			tag: 'nullable|string|min:1|max:25'
		}

		const { errors } = validate(rules, {...req.params, ...req.body})
		if (errors) {
			return res.status(400).json({ message: errors[0] })
		}

		return next()
	} catch (error) {
		console.log(error)
		return res.status(500).json({ message: 'server error' })
	}
}

function deleteMessage(req: Request, res: Response, next: NextFunction) {
	try {
		const rules = {
			id: 'mongoid'
		}

		const { errors } = validate(rules, req.params)
		if (errors) {
			return res.status(400).json({ message: errors[0] })
		}

		return next()
	} catch (error) {
		console.log(error)
		return res.status(500).json({ message: 'server error' })
	}
}

function postPresignUrl(req: Request, res: Response, next: NextFunction) {
	try {
		const rules = {
			chat: 'mongoid',
			files: 'arrayof:string|arrayof:min:5|arrayof:max:300'
		}

		const { errors } = validate(rules, req.body)
		if (errors) {
			return res.status(400).json({ message: errors[0] })
		}

		return next()
	} catch (error) {
		console.log(error)
		return res.status(500).json({ message: 'server error' })
	}
}

const validations = {
	getMessages,
	postMessage,
	postMessageShareReels,
	patchMessage,
	deleteMessage,
	postPresignUrl
}

export default validations
