import { Schema, model, Document, Types } from 'mongoose'

const ReelPhotoSchema = new Schema({
	url: {
		type: String,
		required: true
	},
}, { versionKey: false, _id: false })

export interface IReelPhoto {
	url: string
}

const ReelMetaDataSchema = new Schema({
	comments: String,
	vectorMetaData: {
		type: [Number],
		default: undefined,
	},
	isPositive: Boolean,
	isNegative: Boolean,
	awarenessExercise: {
		type: [String],
		default: undefined,
		enum: ['mindful-breathing', 'noticing-sound', 'body-awareness', 'focus-on-object', 'cloud-watching', 'mindful-walking', 'gratitude-anchor']
	},
	groundingExercise: {
		type: [String],
		default: undefined,
		enum: ['54321-technique', 'body-scan', 'box-breathing', 'object-focus', 'name-label-emotions', 'mindful-walk-scroll', 'name-surroundings']
	},
	isProductivity: Boolean,
	isMeditation: Boolean,
}, { versionKey: false, _id: false })

export interface IReelMetaData {
	comments?: string
	vectorMetaData?: number[]
	isPositive?: boolean
	isNegative?: boolean
	awarenessExercise?: ('mindful-breathing' | 'noticing-sound' | 'body-awareness' | 'focus-on-object' | 'cloud-watching' | 'mindful-walking' | 'gratitude-anchor')[]
	groundingExercise?: ('54321-technique' | 'body-scan' | 'box-breathing' | 'object-focus' | 'name-label-emotions' | 'mindful-walk-scroll' | 'name-surroundings')[]
	isProductivity?: boolean
	isMeditation?: boolean
}

const DynamicEntrySchema = new Schema({
	_id: {
		type: String,
		required: true,
	},
	parent: String,
	label: String,
	cover: String,
	video: {
		type: String,
		required: true
	},
	question: String
}, { versionKey: false })

export interface DynamicReelNested {
	_id?: string | null
	label?: string | null
	cover?: string | null
	video: string
	question?: string | null
	children?: DynamicReelNested[] | null
}

export interface DynamicReelFlat {
	parent?: string | null
	_id?: string | null
	label?: string | null
	cover?: string | null
	video: string
	question?: string | null
}

const ReelCompressionResultSchema = new Schema({
	isCompleted: Boolean,
	errors: Object
}, { versionKey: false, _id: false })

export interface IReelCompressionResult {
	isCompleted?: boolean
	errors?: Record<string, string>
}

const ReelSchema = new Schema(
	{
		user: {
			type: Types.ObjectId,
			ref: 'User',
			required: true,
		},
		chat: {
			type: Types.ObjectId,
			ref: 'Chat',
		},
		message: {
			type: Types.ObjectId,
			ref: 'Message',
		},
		event: {
			type: Types.ObjectId,
			ref: 'Event',
		},
		description: String,
		tag: String,
		subTag: String,
		cover: String,
		video: String,
		compressionResult: {
			type: ReelCompressionResultSchema,
			default: {}
		},
		dynamic: {
			type: [DynamicEntrySchema],
			required: false,
			default: undefined
		},
		vectorIndex: {
			type: [Number],
			default: [],
		},
		hashtags: {
			type: [String],
			default: []
		},
		photos: {
			type: [ReelPhotoSchema],
			default: undefined,
		},
		metaData: ReelMetaDataSchema,
		song: String,
		chips: {
			type: [String],
			enum: ['focus', 'mindfulness', 'reflection'],
			default: undefined
		},
		isAnonymous: Boolean,
		views: {
			type: Number,
			default: 0
		},
		likes: {
			type: Number,
			default: 0
		},
		superLikes: {
			type: Number,
			default: 0
		},
		shares: {
			type: Number,
			default: 0
		},
		comments: {
			type: Number,
			default: 0
		},
		reports: {
			type: Number,
			default: 0
		},
		favorites: {
			type: Number,
			default: 0
		},
	},
	{ versionKey: false, timestamps: true }
)

export interface IReel extends Document {
	_id: Types.ObjectId
	user: Types.ObjectId
	chat?: Types.ObjectId
	message?: Types.ObjectId
	event?: Types.ObjectId
	description?: string
	tag?: string
	subTag?: string
	cover?: string
	video?: string
	compressionResult: IReelCompressionResult
	dynamic?: DynamicReelFlat[]
	chips?: ('focus' | 'mindfulness' | 'reflection')[]
	vectorIndex: number[]
	metaData?: IReelMetaData
	hashtags: string[]
	photos?: IReelPhoto[]
	song?: string
	isAnonymous?: boolean
	views: number
	likes: number
	superLikes: number
	shares: number
	comments: number
	reports: number
	favorites: number
	createdAt: Date
	updatedAt: Date
}

export const Reel = model<IReel>('Reel', ReelSchema, 'reels')
