import { Request, Response, NextFunction } from 'express'
import { validate } from 'super-easy-validator'
import services from '../../utils/services'
import constants from '../../utils/constants'

function getReels(req: Request, res: Response, next: NextFunction) {
	try {
		const rules = {
			limit: 'optional|string|natural',
			dynamic: 'optional|string|boolean',
			chips: 'optional|string|enums:focus,mindfulness,reflection',
		}

		const { errors } = validate(rules, req.query)
		if (errors) {
			return res.status(400).json({ message: errors[0] })
		}

		return next()
	} catch (error) {
		console.log(error)
		return res.status(500).json({ message: 'server error' })
	}
}

function getExploreReels(req: Request, res: Response, next: NextFunction) {
	try {
		const rules = {
			limit: 'optional|string|natural',
			page: 'optional|string|natural',
			chat: 'optional|mongoid',
			tag: 'optional|string|min:1|max:20',
		}

		const { errors } = validate(rules, req.query)
		if (errors) {
			return res.status(400).json({ message: errors[0] })
		}

		return next()
	} catch (error) {
		console.log(error)
		return res.status(500).json({ message: 'server error' })
	}
}

function getUserReels(req: Request, res: Response, next: NextFunction) {
	try {
		const rules = {
			limit: 'optional|string|natural',
			page: 'optional|string|natural',
			userId: 'optional|mongoid'
		}

		const { errors } = validate(rules, {...req.query, ...req.params})
		if (errors) {
			return res.status(400).json({ message: errors[0] })
		}

		return next()
	} catch (error) {
		console.log(error)
		return res.status(500).json({ message: 'server error' })
	}
}

function checkReelId(req: Request, res: Response, next: NextFunction) {
	try {
		const { errors } = validate({id: 'mongoid'}, req.params)
		if (errors) {
			return res.status(400).json({ message: errors[0] })
		}

		return next()
	} catch (error) {
		console.log(error)
		return res.status(500).json({ message: 'server error' })
	}
}

function postPresignUrl(req: Request, res: Response, next: NextFunction) {
	try {
		const rules = {
			id: 'optional|mongoid',
			files: 'arrayof:string|arrayof:min:5|arrayof:max:300'
		}

		const { errors } = validate(rules, req.body)
		if (errors) {
			return res.status(400).json({ message: errors[0] })
		}

		return next()
	} catch (error) {
		console.log(error)
		return res.status(500).json({ message: 'server error' })
	}
}

function postReel(req: Request, res: Response, next: NextFunction) {
	try {
		const rules = {
			id: 'mongoid',
			description: 'optional|string|min:3|max:200',
			tag: 'optional|string|min:1|max:30',
			subTag: 'optional|string|min:1|max:50',
			video: 'optional|url',
			chat: 'optional|mongoid',
			cover: 'optional|url',
			photos: 'optional|array|arrayof:url',
			dynamic: 'optional|object',
			$atleast: 'video|photos|dynamic',
			$atmost: 'video|photos|dynamic',
			hashtags: 'optional|array|arrayof:string|arrayof:min:1|arrayof|max:35',
			isAnonymous: 'optional|boolean',
			socket: `optional|${constants.socketRegex}`
		}

		const { errors } = validate(rules, req.body)
		if (errors) {
			return res.status(400).json({ message: errors[0] })
		}

		if(req.body.dynamic) {
			try {
				req.body.dynamic = services.nestedDynamicToFlat(req.body.dynamic)
				const rules = {
					dynamic: [
						{
							_id: 'string|min:5|max:100',
							label: 'optional|string|min:1|max:200',
							parent: 'optional|string|min:5|max:100',
							cover: 'optional|url',
							video: 'url',
							question: 'optional|string|min:3|max:200'
						}
					]
				}
				const {errors} = validate(rules, req.body)
				if(errors) {
					return res.status(400).json({message: errors[0]})
				}
			} catch (error) {
				return res.status(400).json({message: 'unable to parse dynamic'})
			}
		}

		return next()
	} catch (error) {
		console.log(error)
		return res.status(500).json({ message: 'server error' })
	}
}

function patchReel(req: Request, res: Response, next: NextFunction) {
	try {
		const rules = {
			id: 'mongoid',
			tag: 'optional|string|min:1|max:30',
			subTag: 'optional|string|min:1|max:50',
			description: 'optional|string|min:3|max:200',
			cover: 'optional|url',
			hashtags: 'optional|array|arrayof:string|arrayof:min:1|arrayof|max:35',
			isAnonymous: 'optional|boolean'
		}

		const { errors } = validate(rules, {...req.body, ...req.params})
		if (errors) {
			return res.status(400).json({ message: errors[0] })
		}

		return next()
	} catch (error) {
		console.log(error)
		return res.status(500).json({ message: 'server error' })
	}
}

const validations = {
	getReels,
	getExploreReels,
	getUserReels,
	checkReelId,
	postPresignUrl,
	postReel,
	patchReel
}

export default validations
