import { Request, Response } from 'express'
import { Reward } from '../models'
import helpers from '../../../utils/helpers'
import { IFile } from '../../../utils/types'

async function getRewards(req: Request, res: Response) {
	try {
		interface Payload {
			limit?: string | number
			page?: string | number
			search?: string
			isMeetme?: 'true' | 'false'
			isActive?: 'true' | 'false'
		}
		let { limit = '10', page = '1', search, isMeetme, isActive } = req.query as any as Payload

		limit = +limit
		page = +page
		const skip = (page - 1) * limit

		const query: any = {$and: []}
		if (search) {
			query.$and.push({
				$or: [{ title: RegExp(search, 'i') }, { description: RegExp(search, 'i') }]
			})
		}
		if(isMeetme === 'true') {
			query.isMeetme = true
		} else if(isMeetme === 'false') {
			query.$and.push({
				$or: [{isMeetme: false}, {isMeetme: {$exists: false}}]
			})
		}
		if(isActive === 'true') {
			query.remaining = {$gt: 0}
			query.$and.push({$or: [
				{expiredAt: {$exists: false}},
				{expiredAt: {$gte: new Date()}}
			]})
		}

		const rewards = await Reward
			.find(query)
			.sort({ createdAt: -1 })
			.skip(skip)
			.limit(limit)
		const count = await Reward.countDocuments(query)

		return res.json({ rewards, count })
	} catch (error) {
		console.log(error)
		return res.status(500).json({ message: 'server error' })
	}
}

async function getReward(req: Request, res: Response) {
	try {
		const { id } = req.params

		const reward = await Reward.findById(id)
		if (!reward) {
			return res.status(404).json({ message: 'reward not found' })
		}

		return res.json({ reward })
	} catch (error) {
		console.log(error)
		return res.status(500).json({ message: 'server error' })
	}
}

async function postReward(req: Request, res: Response) {
	interface Payload {
		title: string
		description: string
		total: number | string
		latitude?: number | string
		longitude?: number | string
		locationString?: string
		isMeetme?: 'true' | 'false'
		coupon?: string
		expiredAt?: string | Date
	}
	try {
		let { title, description, total, latitude, longitude, locationString, isMeetme, coupon, expiredAt } = req.body as Payload
		const image = (req.file as IFile)!

		// creating reward
		total = +total
		let data: any = {
			title,
			description,
			total,
			remaining: total,
			isMeetme: isMeetme === 'true',
			coupon: coupon || undefined,
			expiredAt: expiredAt ? new Date(expiredAt) : undefined
		}
		if(typeof latitude === 'string' && typeof longitude === 'string' && typeof locationString === 'string') {
			data.location = {
				type: 'Point',
				coordinates: [+longitude, +latitude]
			}
			data.locationString = locationString
		}
		const reward = new Reward(data)

		// uploading cover image
		const path = `rewards/${reward._id}/${image.filename}`
		const cover = await helpers.uploadFile(image, path)
		if (!cover) {
			return res.status(500).json({ message: 'server error while uploading cover' })
		}
		reward.cover = cover
		helpers.deleteFile(image)

		// saving data
		await reward.save()

		return res.status(201).json({ reward })
	} catch (error) {
		console.log(error)
		return res.status(500).json({ message: 'server error' })
	}
}

async function deleteReward(req: Request, res: Response) {
	try {
		const { id } = req.params

		const reward = await Reward.findById(id)
		if (!reward) {
			return res.status(404).json({ message: 'reward not found' })
		}

		if(reward.remaining < reward.total) {
			return res.status(403).json({message: 'reward can\'t be deleted after its claimed once'})
		}

		await helpers.deleteR2File(reward.cover)
		await reward.deleteOne()

		return res.json({ message: 'reward deleted successfully' })
	} catch (error) {
		console.log(error)
		return res.status(500).json({ message: 'server error' })
	}
}

const controllers = {
	getRewards,
	getReward,
	postReward,
	deleteReward
}

export default controllers
