import { Schema, model, Document, Types } from 'mongoose'
import constants from '../../utils/constants'

const SocialActionSchema = new Schema({
	label: {
		type: String,
		required: true
	},
	subLabel: {
		type: String,
		required: true
	},
	type: {
		type: String,
		enum: constants.socialConnectednessActions.map(e => e.type),
		required: true
	},
	value: {
		type: Number,
		default: 0
	},
	completed: Boolean,
	count: Number
}, {versionKey: false, _id: false})

interface ISocialAction {
	label: string
	subLabel: string
	type: typeof constants.socialConnectednessActions[number]['type']
	value: number
	completed?: boolean
	count?: number
}

const SocialSchema = new Schema({
	score: {
		type: Number,
		default: 0
	},
	day: {
		type: String,
		enum: constants.days,
		required: true,
	},
	actions: {
		type: [SocialActionSchema],
		default: [],
	}
}, {versionKey: false, _id: false})

interface ISocial {
	score: number
	day: typeof constants.days[number]
	actions: ISocialAction[]
}

const MindfulnessActionSchema = new Schema({
	label: {
		type: String,
		required: true
	},
	type: {
		type: String,
		enum: constants.mindfulnessActions.map(e => e.type),
		required: true
	},
	value: {
		type: Number,
		default: 0
	},
	completed: Boolean,
	count: Number,
	startAt: Number
}, {versionKey: false, _id: false})

interface IMindfulnessAction {
	label: string
	type: typeof constants.mindfulnessActions[number]['type']
	value: number
	completed?: boolean
	count?: number
	startAt?: number
}

const JournalSchema = new Schema({
	message: {
		type: String,
		required: true
	},
	at: {
		type: Date,
		default: Date
	}
}, {versionKey: false})

interface IJournal {
	message: string
	at: Date
}

const MindfulnessSchema = new Schema({
	score: {
		type: Number,
		default: 0
	},
	actions: {
		type: [MindfulnessActionSchema],
		default: [],
	},
	journal: JournalSchema
}, {versionKey: false, _id: false})

interface IMindfulness {
	score: number
	actions: IMindfulnessAction[]
	journal?: IJournal
}

const WeeklyScoreSchema = new Schema({
	score: {
		type: Number,
		required: true
	},
	date: {
		type: Date,
		required: true
	},
	day: {
		type: String,
		enum: constants.days,
		required: true
	},
	reference: {
		type: Types.ObjectId,
		required: true,
		ref: 'Score'
	}
}, {versionKey: false, _id: false})

interface IWeeklyScore {
	score: number
	date: Date
	day: typeof constants.days[number]
	reference: Types.ObjectId
}

const ScoreSchema = new Schema(
	{
		user: {
			type: Types.ObjectId,
			ref: 'User',
			required: true,
		},
		date: {
			type: Date,
			default: () => {
				const now = new Date()
				const today = new Date(now.getFullYear(), now.getMonth(), now.getDate())
				return today
			}
		},
		appUsageSeconds: {
			type: Number,
			default: 0
		},
		doomScrollUsageSeconds: {
			type: Number,
			default: 0
		},
		social: {
			type: SocialSchema,
			default: () => {
				const now = new Date()
				const day = constants.days[now.getDay()]
				const actions = structuredClone(constants.socialConnectednessActions.filter(e => e.days.flat().includes(day)))
				return {score: 0, day, actions}
			}
		},
		mindfulness: {
			type: MindfulnessSchema,
			default: () => {
				const actions = structuredClone(constants.mindfulnessActions)
				return {score: 0, actions}
			}
		},
		totalScore: {
			type: Number,
			default: 0
		},
		weeklyScores: {
			type: [WeeklyScoreSchema],
			default: []
		},
		hash: {
			type: Types.ObjectId,
			ref: 'RewardHash'
		},
		reelsSeen: {
			type: Number,
			default: 10
		}
	},
	{ versionKey: false, timestamps: true }
)

export interface IScore extends Document {
	_id: Types.ObjectId
	user: Types.ObjectId
	date: Date
	appUsageSeconds: number
	doomScrollUsageSeconds: number
	social: ISocial
	mindfulness: IMindfulness
	totalScore: number
	weeklyScores: IWeeklyScore[]
	hash?: Types.ObjectId
	reelsSeen: number
	createdAt: Date
	updatedAt: Date
}

export const Score = model('Score', ScoreSchema, 'scores')
