import { Request, Response } from 'express'
import { User } from '../models'
import helpers from '../../../utils/helpers'
import { Types } from 'mongoose'
import services from '../../../utils/services'

async function getUsers(req: Request, res: Response) {
	try {
		interface Payload {
			limit?: string | number
			page?: string | number
			search?: string
			emailVerified?: 'true' | 'false'
			phoneVerified?: 'true' | 'false'
			terminated?: 'true' | 'false'
			'2fa'?: 'true' | 'false'
			verified?: 'true' | 'false'
		}
		let {
			limit = '10',
			page = '1',
			search,
			emailVerified,
			phoneVerified,
			terminated,
			'2fa': tfa,
			verified
		} = req.query as any as Payload

		limit = +limit
		page = +page
		const skip = (page - 1) * limit

		const query: any = { $and: [] }
		if (search) {
			const $or: any = [{ name: RegExp(search, 'i') }, { email: RegExp(search, 'i') }, { phone: RegExp(search, 'i') }]
			query.$and.push({ $or })
			if (Types.ObjectId.isValid(search)) {
				$or.push({ _id: new Types.ObjectId(search) })
			}
		}
		if (emailVerified) {
			if (emailVerified === 'false') {
				query.$and.push({
					$or: [{ 'auth.emailVerified': false }, { 'auth.emailVerified': { $exists: false } }]
				})
			} else {
				query.$and.push({ 'auth.emailVerified': true })
			}
		}
		if (phoneVerified) {
			if (phoneVerified === 'false') {
				query.$and.push({
					$or: [{ 'auth.phoneVerified': false }, { 'auth.phoneVerified': { $exists: false } }]
				})
			} else {
				query.$and.push({ 'auth.phoneVerified': true })
			}
		}
		if (terminated) {
			if (terminated === 'false') {
				query.$and.push({
					$or: [{ terminated: false }, { terminated: { $exists: false } }]
				})
			} else {
				query.$and.push({ terminated: true })
			}
		}
		if (tfa) {
			if (tfa === 'false') {
				query.$and.push({
					$or: [{ 'auth.twoFactorAuthentication': false }, { 'auth.twoFactorAuthentication': { $exists: false } }]
				})
			} else {
				query.$and.push({ 'auth.twoFactorAuthentication': true })
			}
		}
		if (verified) {
			if (verified === 'false') {
				query.$and.push({
					$or: [{ verified: false }, { verified: { $exists: false } }]
				})
			} else {
				query.$and.push({ verified: true })
			}
		}
		const users = await User.find(query, {
			name: 1,
			username: 1,
			dob: 1,
			email: 1,
			phone: 1,
			auth: {
				emailVerified: 1,
				phoneVerified: 1
			},
			photo: 1,
			terminated: 1,
			verified: 1,
			createdAt: 1,
			updatedAt: 1
		}).sort({ createdAt: -1 }).skip(skip).limit(limit)
		const count = await User.countDocuments(query)

		return res.json({ users, count })
	} catch (error) {
		console.log(error)
		return res.status(500).json({ message: 'server error' })
	}
}

async function getUser(req: Request, res: Response) {
	try {
		const { id } = req.params

		const user = await User.findById(id, {
			name: 1,
			username: 1,
			bio: 1,
			url: 1,
			school: 1,
			courses: 1,
			dob: 1,
			email: 1,
			phone: 1,
			auth: {
				twoFactorAuthentication: 1,
				emailVerified: 1,
				phoneVerified: 1
			},
			photo: 1,
			terminated: 1,
			verified: 1,
			purposes: 1,
			studies: 1,
			categories: 1,
			createdAt: 1,
			updatedAt: 1
		})
		if (!user) {
			return res.status(404).json({ message: 'user not found' })
		}

		return res.json({ user })
	} catch (error) {
		console.log(error)
		return res.status(500).json({ message: 'server error' })
	}
}

async function patchUser(req: Request, res: Response) {
	interface Payload {
		verified?: boolean
	}
	try {
		const { id } = req.params
		const {verified} = req.body as Payload

		const user = await User.findById(id)
		if (!user) {
			return res.status(404).json({ message: 'user not found' })
		}

		user.verified = verified ?? user.verified
		await user.save()

		return res.json({ message: 'updated successfully' })
	} catch (error) {
		console.log(error)
		return res.status(500).json({ message: 'server error' })
	}
}

async function deleteUser(req: Request, res: Response) {
	try {
		const { id } = req.params

		const user = await User.findById(id)
		if (!user) {
			return res.status(404).json({ message: 'user not found' })
		}

		if (user.terminated) {
			user.terminated = undefined
			await helpers.sendOauthGmail(
				user.email,
				'Your account has been reactivated',
				'Good news!\nAfter careful review, we are pleased to inform you that your account has been reactivated.'
			)
		} else {
			user.terminated = true
			await helpers.sendOauthGmail(
				user.email,
				'Your account has been terminated',
				'Your account has been terminated due to a violation of our Terms and Conditions and in accordance with our Privacy Policy.'
			)
			services.socketNotifyUser({
				userId: `${user._id}`,
				event: 'account-terminated',
				data: { message: 'Your account has been terminated due to a violation of our Terms and Conditions and in accordance with our Privacy Policy.' },
			})
		}
		user.auth.loggedinDevices = [] as any
		await user.save()

		return res.json({
			message: `user's ${user.terminated ? 'terminated' : 'reactivated'} successfully`,
			terminated: user.terminated
		})
	} catch (error) {
		console.log(error)
		return res.status(500).json({ message: 'server error' })
	}
}

const controllers = {
	getUsers,
	getUser,
	patchUser,
	deleteUser
}

export default controllers
