import { Admin } from '../api/admin/models'
import { ILoggedinDevice, IUser, User } from '../api/users/models'
import constants from './constants'
import helpers from './helpers'
import { DynamicReelNested, DynamicReelFlat, IReel, Reel } from '../api/reels/models'
import { AIReelPromptResponse, ISocket, NotificationPayload, PushNotificationPayload, ReelData, ScheduleEvent, SocketErrorEvent, SocketEvent, SocketNotificationPayload } from './types'
import { INotification, Notification } from '../api/notifications/models'
import { Chat } from '../api/chats/models'
import config from './config'
import { IScore, Score } from '../api/scores/models'
import { RewardHash } from '../api/reward-hashes/models'
import { Reward } from '../api/rewards/models'
import { Types } from 'mongoose'
import fs from 'fs/promises'
import { Blocking } from '../api/blockings/models'

async function createAdmin(name: string, email: string, password: string, phone?: string) {
	const hash = helpers.getHash(password)
	const admin = await Admin.create({
		name,
		email,
		phone,
		auth: { password: hash },
	})
	console.log(admin)
}

// createAdmin('Test Admin', 'test@example.com', '123456')

async function deleteOldUnverifiedUsers() {
	console.log('🕒  Deleting unverified users!')

	try {
		const date = new Date()
		date.setDate(date.getDate() - 7)

		const result = await User.deleteMany({
			$or: [
				{ 'auth.emailVerified': { $exists: false } },
				{ 'auth.emailVerified': false }
			],
			createdAt: { $lt: date }
		})

		console.log(`✅  Unverified users deleted: ${result.deletedCount}`)
	} catch (error) {
		console.error('❌ Error while deleting unverified users:', error)
	}
}

async function deleteDeactivatedUsers() {
	console.log('🕒  Deleting old deactivated users!')

	try {
		const date = new Date()
		date.setDate(date.getDate() - 7)

		const users = await User.find({
			$or: [
				{ terminated: { $exists: false } },
				{ terminated: false }
			],
			deactivatedAt: { $lt: date }
		})
		for (const user of users) {
			// TODO: set user fields as undefined
			// TODO: set all those user field in user's "data" field
			// TODO: terminate the user account
		}

		console.log(`✅  Old deactivated users deletion completed!`)
	} catch (error) {
		console.error('❌ Error while deleting deactivated users:', error)
	}
}

async function deleteOldNotifications() {
	console.log('🕒  Deleting old notifications!')

	try {
		const date = new Date()
		date.setDate(date.getDate() - 7)

		const result = await Notification.deleteMany({ at: { $lt: date } })

		console.log(`✅  Old notifications deleted count: ${result.deletedCount}`)
	} catch (error) {
		console.error('❌ Error while deleting old notifications:', error)
	}
}

async function sendOtp(user: IUser, sendTo: 'phone' | 'email' = 'phone') {
	try {
		// don't send otp if already send within delay time, in this case just update expiry
		if (sendTo === 'phone' && user.auth.lastPhoneOtpSentAt) {
			const lastSentAt = new Date(user.auth.lastPhoneOtpSentAt)
			lastSentAt.setSeconds(lastSentAt.getSeconds() + constants.minPhoneOtpResendDelay)
			if (lastSentAt >= new Date()) {
				user.auth.lastPhoneOtpSentAt = new Date()
				const expiry = new Date()
				expiry.setMinutes(expiry.getMinutes() + constants.phoneOtpExpiry)
				user.auth.phoneOtpExpiry = expiry
				await user.save()
				return
			}
		}
		if (sendTo === 'email' && user.auth.lastEmailOtpSentAt) {
			const lastSentAt = new Date(user.auth.lastEmailOtpSentAt)
			lastSentAt.setSeconds(lastSentAt.getSeconds() + constants.minEmailOtpResendDelay)
			if (lastSentAt >= new Date()) {
				user.auth.lastEmailOtpSentAt = new Date()
				const expiry = new Date()
				expiry.setMinutes(expiry.getMinutes() + constants.emailOtpExpiry)
				user.auth.emailOtpExpiry = expiry
				await user.save()
				return
			}
		}

		// now send the otp if its expired or not exist
		const otp = helpers.getRandomOtp()
		const expiry = new Date()
		if (sendTo === 'phone') {
			expiry.setMinutes(expiry.getMinutes() + constants.phoneOtpExpiry)
			user.auth.phoneOtp = otp
			user.auth.phoneOtpExpiry = expiry
			user.auth.lastPhoneOtpSentAt = new Date()
			await helpers.sendSms(user.phone!, `Your OTP is ${otp}`)
		} else {
			expiry.setMinutes(expiry.getMinutes() + constants.emailOtpExpiry)
			user.auth.emailOtp = otp
			user.auth.emailOtpExpiry = expiry
			user.auth.lastEmailOtpSentAt = new Date()
			await helpers.sendOauthGmail(user.email, 'OTP Verification', `Your OTP is ${otp}`)
		}
		await user.save()
	} catch (error) {
		console.log(error)
	}
}

async function checkAndResetOtp(user: IUser, otp: string, sendTo: 'phone' | 'email' = 'phone', reset: boolean = false): Promise<{ status?: number, message?: string, error?: any }> {
	try {
		const now = new Date()
		if (sendTo === 'phone') {
			if (!user.auth.phoneOtpExpiry) {
				return { status: 404, message: 'OTP not found' }
			}
			if (user.auth.phoneOtpExpiry < now) {
				return { status: 410, message: 'OTP expired' }
			}
			if (user.auth.phoneOtp !== otp) {
				return { status: 401, message: 'OTP verification failed' }
			}
			if (reset) {
				user.auth.phoneOtp = undefined
				user.auth.phoneOtpExpiry = undefined
				user.auth.phoneVerified = true
				await user.save()
			}
		} else {
			if (!user.auth.emailOtpExpiry) {
				return { status: 404, message: 'OTP not found' }
			}
			if (user.auth.emailOtpExpiry < now) {
				return { status: 410, message: 'OTP expired' }
			}
			if (user.auth.emailOtp !== otp) {
				return { status: 401, message: 'OTP verification failed' }
			}
			if (reset) {
				user.auth.emailOtp = undefined
				user.auth.emailOtpExpiry = undefined
				user.auth.emailVerified = true
				await user.save()
			}
		}
		return {}
	} catch (error) {
		return { error }
	}
}

async function pushNotifyUser(payload: PushNotificationPayload) {
	try {
		let { user, loggedinDevice, title, body, event, data = {}, deviceType = 'all' } = payload
		const removals: ILoggedinDevice[] = []
		for (const device of user.auth.loggedinDevices) {
			if (!device.fcmToken) {
				continue
			}
			if (deviceType === 'loggedin' && `${device._id}` !== `${loggedinDevice}`) {
				continue
			}
			if (deviceType === 'others' && `${device._id}` === `${loggedinDevice}`) {
				continue
			}
			data = { ...data, event }
			const error = await helpers.sendPushNotification(device.fcmToken, title, body, data)
			if (error) {
				removals.push(device)
			}
		}
		if (removals.length) {
			user.auth.loggedinDevices = user.auth.loggedinDevices.filter(e => !removals.some(r => r.fcmToken === e.fcmToken))
			await user.save()
		}
	} catch (error) {
		console.log(error)
	}
}

function socketNotifyUser(payload: SocketNotificationPayload) {
	try {
		const { userId, event, data, currentSocket, socketType = 'all' } = payload
		const sockets = constants.sockets[userId]
		if (!sockets) {
			return
		}
		for (const socket of sockets) {
			if (socketType === 'current' && currentSocket && `${socket.id}` !== currentSocket) {
				continue
			}
			if (socketType === 'others' && socket.id === currentSocket) {
				continue
			}
			socket.emit('message', { event, data })
			const user = socket.data.user as IUser
			console.log(`💬 socket message sent to ${user.username ?? user.email}`)
			console.dir({ event, data }, { depth: null })
		}
	} catch (error) {
		console.log(error)
	}
}

function socketErrorNotifyUser(socket: ISocket, event: SocketErrorEvent, data: Record<string, any>) {
	try {
		socket.emit('error', { event, data })
		const user = socket.data.user as IUser
		console.log(`❌ socket error message sent to ${user.username ?? user.email}`)
		console.dir({ event, data }, { depth: null })
	} catch (error) {
		console.log(error)
	}
}

/**
 * User only requires _id and auth field
 */
async function notifyUser(payload: NotificationPayload) {
	try {
		let { user, data = {}, title, description, event, from, comment, connection, follower, like, superLike, reel, status, chat, request, meetMeEvent, rewardHash } = payload
		const notification = await Notification.create({
			user: user._id,
			data,
			title,
			description,
			event,
			from,
			comment,
			connection,
			follower,
			like,
			superLike,
			reel,
			status,
			chat,
			request,
			meetMeEvent,
			rewardHash
		})
		data = {
			...data,
			notification: `${notification._id}`,
			...(from ? { from: `${from}` } : {}),
			...(comment ? { comment: `${comment}` } : {}),
			...(connection ? { connection: `${connection}` } : {}),
			...(follower ? { follower: `${follower}` } : {}),
			...(like ? { like: `${like}` } : {}),
			...(superLike ? { superLike: `${superLike}` } : {}),
			...(reel ? { reel: `${reel}` } : {}),
			...(status ? { status: `${status}` } : {}),
			...(request ? { request: `${request}` } : {}),
			...(chat ? { chat: `${chat}` } : {}),
			...(meetMeEvent ? { meetMeEvent: `${meetMeEvent}` } : {}),
			...(rewardHash ? { rewardHash: `${rewardHash}` } : {}),
		};

		(async () => {
			try {
				await pushNotifyUser({
					user,
					title,
					body: description,
					event,
					data,
				})
			} catch (error) {
				console.log(error)
			}
		})()

		return notification as unknown as INotification
	} catch (error) {
		console.log(error)
	}
}

function nestedDynamicToFlat(dynamic: DynamicReelNested, list: DynamicReelFlat[] = [], parent?: DynamicReelNested) {
	let { _id, label, cover, video, question, children } = dynamic
	if (!_id || (typeof _id === 'string' && !/^[0-9a-f]{24}$/.test(_id))) {
		_id = Date.now() + '-' + Math.random().toString().substring(2)
		dynamic._id = _id
	}
	const entry: DynamicReelFlat = { _id, label, cover, video, question, ...(parent ? { parent: parent._id } : {}) }
	list.push(entry)
	if (children?.length) {
		for (const child of children) {
			nestedDynamicToFlat(child, list, dynamic)
		}
	}
	return list
}

function flatDynamicToNested(serialized: DynamicReelFlat[]): DynamicReelNested {
	const insertEntryInDynamic = (dynamic: DynamicReelNested | undefined, entry: DynamicReelFlat, root: DynamicReelNested): DynamicReelNested => {
		const { _id, label, cover, video, question, parent } = entry
		if (!dynamic) {
			dynamic = { _id, label, cover, video, question, children: [] }
			return root ?? dynamic
		}
		if (!dynamic.children) {
			dynamic.children = []
		}
		if (parent && dynamic._id === parent) {
			dynamic.children.push({ _id, label, cover, video, question })
		}
		for (const child of dynamic.children) {
			root = insertEntryInDynamic(child, entry, root)
		}
		if (!dynamic.children.length) {
			dynamic.children = undefined
		}
		return root
	}
	let dynamic: DynamicReelNested | undefined = undefined
	for (const entry of serialized) {
		if (entry.parent) {
			dynamic = insertEntryInDynamic(dynamic, entry, dynamic!)
		} else {
			const { _id, label, cover, video, question } = entry
			dynamic = { _id, label, cover, video, question, children: [] }
		}
	}
	return dynamic!
}

async function createLoggedinDevice(user: IUser, platform?: 'android' | 'ios' | 'web', fcmToken?: string, remember = true) {
	let devices = user.auth.loggedinDevices
	if (devices.length >= constants.maxLoggedinDevices) {
		// remove the device with oldest accessed
		devices = devices.sort((a, b) => b.lastAccessedAt.getTime() - a.lastAccessedAt.getTime())
		devices = devices.slice(0, -1)
	}
	const token = helpers.getToken(user as IUser, remember)
	const tokenHash = helpers.getHash(token)
	// @ts-ignore
	devices.push({ platform, tokenHash, fcmToken })
	user.auth.loggedinDevices = devices
	await user.save()
	return token
}

async function socketNotifyOnUserPresenceUpdated(user: IUser, status: 'online' | 'offline') {
	interface Payload {
		user: PayloadUser
		chats: string[]
		chat?: string
	}
	interface PayloadUser {
		_id: string
		name?: string
		username?: string
		photo?: string
		isOnline?: boolean
		isManuallyOffline?: boolean
	}
	try {
		// fetch all distinct chat participants where this user belongs
		const chats = await Chat.find({ 'participants.user': user._id }, { isGroup: 1, 'participants.user': 1 })
			.populate({
				path: 'participants.user',
				select: { name: 1, username: 1, photo: 1 }
			})
		const people = <Record<string, Payload>>{}
		let id: string
		for (const chat of chats) {
			for (const person of chat.participants) {
				id = `${(person.user as IUser)._id}`
				if (id === `${user._id}`) {
					continue
				}
				if (!(id in people)) {
					people[id] = {
						user: {
							_id: `${user._id}`,
							name: user.name,
							username: user.username,
							photo: user.photo,
						},
						chats: []
					}
				}
				people[id].chats.push(`${chat._id}`)
				if(!chat.isGroup) {
					people[id].chat = chat._id.toString()
				}
			}
		}
		for (const userId in people) {
			people[userId].chats = [...new Set(people[userId].chats)]
		}

		// emit socket events to those users
		const event: SocketEvent = status === 'offline' ? 'user-offline' : 'user-online'
		for (const userId in people) {
			socketNotifyUser({
				userId,
				event,
				data: people[userId]
			})
		}
	} catch (error) {
		console.log(error)
	}
}

async function runAiPrompt(userPrompt: string, systemPrompt: string) {
	try {
		const response = await config.openai.chat.completions.create({
			model: 'gpt-4o-mini',
			messages: [
				{ role: 'system', content: systemPrompt },
				{ role: 'user', content: userPrompt }
			]
		})
		return response.choices[0].message.content
	} catch (error) {
		console.log(error)
	}
}

async function analyzeReelByAi(systemPrompt: string, reelData: ReelData) {
	try {
		const content: any = []
		if (reelData.video) {
			content.push({ type: 'input_file', file_url: reelData.video })
		} else if (reelData.photos) {
			for (const image of reelData.photos) {
				content.push({ type: 'input_file', file_url: image })
			}
		} else if (reelData.dynamic) {
			for (const video of reelData.dynamic) {
				content.push({ type: 'input_file', file_url: video })
			}
		}
		const response = await config.openai.responses.create({
			model: 'gpt-4o-mini',
			input: [
				{ role: 'system', content: systemPrompt },
				{ role: 'user', content }
			]
		})
		const json = JSON.parse(response.output_text) as AIReelPromptResponse
		return json
	} catch (error) {
		console.log(error)
	}
}

async function updateScore(score: IScore) {
	try {
		if (score.totalScore === 1) {
			return
		}

		// recalculate social score and save it
		const social = score.social
		let count = social.actions.length
		let completedCount = social.actions.filter(e => e.completed).length
		let totalValues = social.actions.reduce((acc, e) => acc + e.value, 0.0)
		social.score = totalValues / count
		social.score = +(social.score).toFixed(2)
		if (completedCount === count) {
			social.score = 1
		}

		// recalculate mindfulness score and save it
		const mindfulness = score.mindfulness
		count = mindfulness.actions.length
		completedCount = mindfulness.actions.filter(e => e.completed).length
		totalValues = mindfulness.actions.reduce((acc, e) => acc + e.value, 0.0)
		mindfulness.score = totalValues / count
		mindfulness.score = +(mindfulness.score).toFixed(2)
		if (completedCount === count) {
			mindfulness.score = 1
		}

		score.totalScore = (social.score + mindfulness.score) / 2
		score.totalScore = +score.totalScore.toFixed(2)
		score.totalScore = Math.min(score.totalScore, 1)
		await score.save()

		// send socket notification
		const _mindfulness = JSON.parse(JSON.stringify(score.mindfulness))
		delete _mindfulness.journal
		services.socketNotifyUser({
			userId: `${score.user}`,
			event: 'score-updated',
			data: {
				date: score.date,
				reelsSeen: score.reelsSeen,
				totalScore: score.totalScore,
				social: JSON.parse(JSON.stringify(score.social)),
				mindfulness: _mindfulness
			}
		})

		if (score.totalScore === 1) {
			const day = constants.days[score.date.getDay()]
			if (day !== 'wednesday' && day !== 'sunday') {
				return
			}
			const previousScores = score.weeklyScores
			let isRewardEligible = false

			// checking if the day is wednesday or thursday, if yes, then all previous days scores must be all completed
			const monday = previousScores.find(e => e.day === 'monday')?.score === 1
			const tuesday = previousScores.find(e => e.day === 'tuesday')?.score === 1
			const wednesday = previousScores.find(e => e.day === 'wednesday')?.score === 1
			const thursday = previousScores.find(e => e.day === 'thursday')?.score === 1
			const friday = previousScores.find(e => e.day === 'friday')?.score === 1
			const saturday = previousScores.find(e => e.day === 'saturday')?.score === 1
			if (day === 'wednesday' && monday && tuesday) {
				isRewardEligible = true
			}
			if (day === 'sunday' && monday && tuesday && wednesday && thursday && friday && saturday) {
				isRewardEligible = true
			}
			if (!isRewardEligible) {
				return
			}

			// now creating reward for that user
			const user = await User.findById(score.user)
			if (!user) {
				return
			}

			// fetching unique profile rewardIds from reward hashes which already exist for this user
			const rewardIds = (await RewardHash.find({ user: score.user }, { reward: 1 })).map(e => e.reward)
			// now fetching any one reward where _id don't includes any from rewardIds
			rewardIds.forEach(e => console.log(`${e}`))
			const query: any = {
				_id: { $nin: rewardIds },
				remaining: { $gt: 0 },
				$and: [
					{
						$or: [
							{ isMeetme: false },
							{ isMeetme: { $exists: false } }
						]
					},
					{
						$or: [
							{ expiredAt: { $gt: new Date() } },
							{ expiredAt: { $exists: false } }
						]
					},
				]
			}
			if (user.location) {
				query.$and.push({
					$or: [
						{
							location: {
								$geoWithin: {
									$centerSphere: [
										user.location.coordinates, // [lng, lat]
										constants.locationDistance / constants.earthRadius // meters → radians
									]
								}
							}
						},
						{ location: { $exists: false } }
					]
				})
			}
			const reward = await Reward.findOne(query)
			if (!reward) {
				return
			}

			// creating reward hash for this reward
			const rewardHash = await RewardHash.create({
				user: user._id,
				reward: reward._id,
				hash: helpers.getHash(`${user._id}:${reward._id}`),
				score: score._id,
			})
			score.hash = rewardHash._id
			await score.save()

			// finally notify user
			await services.notifyUser({
				title: 'Reward Unlocked',
				description: `Congrats ${user.name?.split(' ')[0]}! you have unlocked a new reward`,
				event: 'reward-unlocked',
				user: user as IUser,
				rewardHash: rewardHash._id
			})
		}
	} catch (error) {
		console.error(error)
	}
}

async function getOrCreateTodayScore(userId: string | Types.ObjectId) {
	try {
		const now = new Date()
		const today = new Date(now.getFullYear(), now.getMonth(), now.getDate())

		let score = await Score.findOne({ date: today, user: userId })
		if (score) {
			return score
		}

		const weekAgo = new Date(today)
		weekAgo.setDate(today.getDate() - 7)
		const scores = await Score
			.find({ date: { $gte: weekAgo }, user: userId })
			.sort({ createdAt: -1 })

		score = await Score.create({
			user: userId,
			date: today,
			weeklyScores: scores.map(e => ({
				score: e.totalScore,
				date: e.date,
				day: constants.days[e.date.getDay()],
				reference: e._id
			}))
		})

		return score
	} catch (error) {
		console.error(error)
	}
}

function enqueueVideoCompression(
	inputUrl: string,
	message: string,
	onDone: (err: Error | undefined, outputUrl: string | undefined) => Promise<void>
) {
	const getCompressedUrls = (url: string) => {
		const inputPath = 'temp/' + url.split('/').at(-1)
		const lastDot = url.lastIndexOf('.')
		let u = ''
		if (lastDot === -1) {
			u = url + '-compressed'
		} else {
			const base = url.substring(0, lastDot)
			const ext = url.substring(lastDot)
			u = `${base}-compressed${ext}`
		}
		const outputFile = u.split('/').at(-1)
		const outputPath = 'temp/' + outputFile
		const outputCloudFilePath = u.substring(u.indexOf('reels/'))
		return { outputUrl: u, outputFile, outputCloudFilePath, outputPath, inputPath }
	}
	return config.compressionQueue.add(async () => {
		try {
			constants.compressionProcessingCount++
			console.log(message)
			// downloading video
			const inputFile = inputUrl.split('/').at(-1)
			const inputPath = 'temp/' + inputFile
			await helpers.downloadFile(inputUrl, inputPath)
			// generate output path, and pass it to the function
			const { outputUrl, outputFile, outputCloudFilePath, outputPath } = getCompressedUrls(inputUrl)
			await helpers.compressVideo(inputPath, outputPath)
			// check if generated file size is smaller or even more than previous size
			const [inputStat, outputStat] = await Promise.all([fs.stat(inputPath), fs.stat(outputPath)])
			if (outputStat.size >= inputStat.size) {
				throw new Error('Unable to compress even more, file already compressed')
			}
			// upload the file to cloudflare R2 with same path
			const fileBuffer = await fs.readFile(outputPath)
			await helpers.uploadFile(fileBuffer, outputCloudFilePath)
			await onDone(undefined, outputUrl)
		} catch (error) {
			// handling onDone in error case
			console.log(error)
			try {
				await onDone(error as Error, undefined)
			} catch (error) {
				console.log(error)
			}
		} finally {
			// deleting file
			try {
				const { outputPath, inputPath } = getCompressedUrls(inputUrl)
				await fs.unlink(outputPath)
				await fs.unlink(inputPath)
			} catch (error) {
				console.log(error)
			}
			constants.compressionProcessingCount--
		}
	})
}

async function addReelToCompressionQueue(reel: IReel) {
	try {
		const hasCompressedUrl = (url: string) => {
			const lastDot = url.lastIndexOf('.')
			if (lastDot === -1) return url.endsWith('-compressed')
			const base = url.substring(0, lastDot)
			return base.endsWith('-compressed')
		}
		if(!reel.dynamic && !reel.video) {
			return
		}
		if (reel.compressionResult.isCompleted) {
			return
		}
		if (`${reel._id}` in constants.reelsInCompression) {
			return
		}
		constants.reelsInCompression[`${reel._id}`] = true
		console.log(`🧱 Adding reel ${reel._id} to compression queue`)
		if (reel.dynamic) {
			for (let i = 0; i < reel.dynamic.length; i++) {
				const dynamic = reel.dynamic[i]
				if (hasCompressedUrl(dynamic.video)) {
					continue
				}
				enqueueVideoCompression(dynamic.video, `🧱 Reel ${reel._id} dynamic.${i} compression started`, async (err, outputUrl) => {
					try {
						if (err) {
							reel.compressionResult.errors ||= {}
							reel.compressionResult.errors[`dynamic.${i}`] = err.toString()
						}
						if (outputUrl) {
							dynamic.video = outputUrl
						}
						await reel.save()
						if (!reel.video) {
							reel.compressionResult.isCompleted = true
							await reel.save()
							delete constants.reelsInCompression[`${reel._id}`]
							console.log(`✅ reel ${reel._id} dynamic.${i} compression completed`)
						}
					} catch (error) {
						console.log(error)
						if (!reel.video) {
							delete constants.reelsInCompression[`${reel._id}`]
						}
					} finally {
					}
				})
			}
		}
		if (reel.video && !hasCompressedUrl(reel.video)) {
			enqueueVideoCompression(reel.video, `🧱 Reel ${reel._id} video compression started`, async (err, outputUrl) => {
				try {
					if (err) {
						reel.compressionResult.errors ||= {}
						reel.compressionResult.errors['video'] = err.toString()
					}
					if (outputUrl) {
						reel.video = outputUrl
					}
					await reel.save()
					reel.compressionResult.isCompleted = true
					await reel.save()
					delete constants.reelsInCompression[`${reel._id}`]
					console.log(`✅ reel ${reel._id} video compression completed`)
				} catch (error) {
					console.log(error)
					delete constants.reelsInCompression[`${reel._id}`]
					console.log(`✅ reel ${reel._id} video compression completed`)
				}
			})
		}
		if (!reel.dynamic && !(reel.video && !hasCompressedUrl(reel.video))) {
			reel.compressionResult.isCompleted = true
			await reel.save()
			delete constants.reelsInCompression[`${reel._id}`]
			console.log(`✅ reel ${reel._id} compression completed`)
		}
	} catch (error) {
		console.log(error)
	}
}

async function addUncompressedReelsToQueue() {
	try {
		if(!constants.compressionEnabled) {
			return
		}
		if (config.compressionQueue.size >= constants.maxCompressionQueueSize) {
			return
		}
		const reels = await Reel.find({
			$or: [
				{ 'compressionResult.isCompleted': false },
				{ 'compressionResult.isCompleted': { $exists: false } },
			]
		}).sort({ createdAt: 1 }).limit(100)
		for (const reel of reels) {
			await addReelToCompressionQueue(reel)
			if (config.compressionQueue.size >= constants.maxCompressionQueueSize) {
				break
			}
		}
	} catch (error) {
		console.log(error)
	}
}

async function clearTempFolder() {
	try {
		const files = await fs.readdir('temp')
		for (const file of files) {
			await fs.unlink(`temp/${file}`)
		}
	} catch (err) { }
}

async function isUserBlocked(userId1: Types.ObjectId, userId2: Types.ObjectId) {
	const blocking = await Blocking.findOne({
		$or: [
			{user: userId1, blockedBy: userId2},
			{blockedBy: userId1, user: userId2},
		]
	})
	return !!blocking
}

async function createScheduler(event: ScheduleEvent, at: Date, data: Record<string, any>) {
	try {
		const id = `job-${event}-${Date.now()}-${Math.random().toString().split('.').at(-1)}`
		await helpers.createEventBridgeSchedule(id, at, {id, event, data})
		console.log('⏰ Schedule created:', id)
		return id
	} catch (error) {
		console.log('Error occurred while create schedule')
		console.log(error)
	}
}

// config.compressionQueue.addListener('idle', addUncompressedReelsToQueue)
addUncompressedReelsToQueue()

const services = {
	createAdmin,
	sendOtp,
	checkAndResetOtp,
	deleteOldUnverifiedUsers,
	deleteDeactivatedUsers,
	pushNotifyUser,
	nestedDynamicToFlat,
	notifyUser,
	flatDynamicToNested,
	createLoggedinDevice,
	deleteOldNotifications,
	socketNotifyUser,
	socketErrorNotifyUser,
	socketNotifyOnUserPresenceUpdated,
	runAiPrompt,
	analyzeReelByAi,
	updateScore,
	getOrCreateTodayScore,
	enqueueVideoCompression,
	addReelToCompressionQueue,
	addUncompressedReelsToQueue,
	clearTempFolder,
	isUserBlocked,
	createScheduler
}

export default services
